/***
 * additivesynthesizer.h
 *
 * The engine of an additive synthesizer
 * Created by Jussi Pekonen
 ***/

/* Header indentification */
#ifndef _ADDITIVESYNTHESIZER_
#define _ADDITIVESYNTHESIZER_


/* Check if we use the code in C++ */
#ifdef __cplusplus
extern "C" {
#endif
		
/* Required libraries */
#include <math.h>
#include <stdlib.h>

/**
 * Function for initializing the sine table (one cycle)
 *
 * @param N The table size
 * @return Pointer to the beginning of the created table
 **/
float* additivesynthesizer_tableinit(const int N) {
	// Allocate the memory of the sine table
	float* sinetable = (float*) malloc(N*sizeof(float));
	while (sinetable == NULL)
		sinetable = (float*) malloc(N*sizeof(float));
	// Initialize the table
	int k;
	for (k = 0; k < N; k++)
		sinetable[k] = sin((2.0*M_PI*k)/N);
	return sinetable;
}
	
/**
 * Function for destructing the sine table
 *
 * @param sinetable Pointer to the beginning of the table to be destructed
 **/
void additivesynthesizer_tablefree(float* sinetable) {
	// Free the allocated memory
	free(sinetable);
}
	
/**
 * Function for computing a value from the sine table
 * Uses linear interpolation
 *
 * @param phase The phase of the oscillator
 * @param sinetable Pointer to the beginning of the sine table
 * @param tablesize Size of the sine table
 * @return Value of the sine function at given phase
 **/
float additivesynthesizer_tablelookup(const float phase, const float* sinetable, const int tablesize) {
	// The phasor value as a table index
	float tableindex = tablesize*phase;
	// The fractional part
	float frac_shift = tableindex -  (int) tableindex;
	// The lower bound
	int l_tableindex = ((int) tableindex) % tablesize;
	// The upper bound
	int u_tableindex = (l_tableindex + 1) % tablesize;
	// Interpolate
	return sinetable[l_tableindex]*(1 - frac_shift) + sinetable[u_tableindex]*frac_shift;
}
	
/**
 * Function for generating the tone
 *
 * @param sinetable Pointer to the sine table
 * @param tablesize Size of the sine table
 * @param compcount The number of components
 * @param freqs The components frequencies
 * @param phases The current components phases (in the same order as the frequencies)
 * @param amps The components amplitudes (in the same order as the frequencies)
 * @param Fs Sampling rate
 * @param ampthreshold Lower limit for the amplitudes
 * @param maxfreq Maximum limit for the frequencies
 * @return The sample value of that tone
 **/
float additivesynthesizer(const float* sinetable, const int tablesize, const int compcount, const float* freqs, float* phases, const float* amps, const float Fs, const float ampthreshold, const float maxfreq) {
	// Component counter
	int k;
	// The output value
	float output = 0.0;
	// Loop through the components
	for (k = 0; k < compcount; k++) {
		// Synthesize only the components that make sense
		if ((freqs[k] <= maxfreq) && (amps[k] >= ampthreshold)) {
			// Add to the output
			output += amps[k]*additivesynthesizer_tablelookup(phases[k], sinetable, tablesize);
		}
		// Step the phases
		phases[k] += freqs[k]/Fs;
	}
	// Return the output
	return output;
}

#ifdef __cplusplus
}
#endif

#endif
