/***
 * amplitudeenvelope.h
 *
 * The engine of an amplitude envelope generator
 * Created by Jussi Pekonen
 ***/

/* Header indentification */
#ifndef _AMPLITUDEENVELOPE_
#define _AMPLITUDEENVELOPE_

/* Check if we use the code in C++ */
#ifdef __cplusplus
extern "C" {
#endif

/* Global variables */
float AMPLITUDE_ENVELOPE_ATTACK = 1.0; // The state value for the attack phase
float AMPLITUDE_ENVELOPE_DECAY = 2.0; // The state value for the decay phase
float AMPLITUDE_ENVELOPE_SUSTAIN = 3.0; // The state value for the sustain phase
float AMPLITUDE_ENVELOPE_RELEASE = 4.0; // The state value for the release phase
float AMPLITUDE_ENVELOPE_OFF = 5.0; // The state value for the no-sound phase
		
/**
 * Function that generates the attack phase
 *
 * @param attacktime The attack time in samples
 * @param state Pointer to the previous amplitude value
 * @return The new amplitude value
 **/
float amplitudeenvelopeattack(const float attacktime, float* state) {
	// Compute the amplitude
	float amplitude = (*state)*(2.0 - (*state));
	// Update the amplitude envelope value state
	if (attacktime) {
		(*state) += 1.0/attacktime;
	}
	else {
		(*state) = 1;
	}
	// Return a smoothed envelope value
	return amplitude;
}

/**
 * Function that generates the decay phase
 *
 * @param decaytime The decay time constant in samples
 * @param sustainlevel The sustain level
 * @param state Pointer to the previous amplitude value
 * @return The new amplitude value
 **/
float amplitudeenvelopedecay(const float decaytime, const float sustainlevel, float* state) {
	// Compute the amplitude
	float amplitude = sustainlevel + (1 - sustainlevel)*(*state)*(*state);
	// Update the amplitude envelope value state
	if (decaytime) {
		(*state) = (((*state) - 1.0/decaytime) < 0.0) ? 0.0 : (*state) - 1.0/decaytime;
	}
	else {
		(*state) = 0.0;
	}
	// Return a smoothed envelope value
	return amplitude;
}
	
/**
 * Function that generates the release phase
 *
 * @param releasetime The release time constant in samples
 * @param state Pointer to the previous amplitude value
 * @return The new amplitude value
 **/
float amplitudeenveloperelease(const float releasetime, float* state) {
	// Compute the amplitude
	float amplitude = (1.0 - (*state))*(1.0 - (*state));
	// Update the amplitude envelope value state
	if (releasetime) {
		(*state) = (((*state) + 1.0/releasetime) > 1.0) ? 1.0 : (*state) + 1.0/releasetime;
	}
	else {
		(*state) = 1.0;
	}
	// Return a smoothed envelope value
	return amplitude;
}

/**
 * Function that generates the amplitude envelope
 *
 * @param attacktime The attack duration in samples
 * @param decaytime The decay time constant in samples
 * @param sustainlevel The sustain level
 * @param releasetime The release time constant in samples
 * @param prevamplitude Pointer to the previous amplitude value
 * @param stage Pointer to the stage of the envelope
 * @return The new amplitude value
 **/
float amplitudeenvelope(const float attacktime, const float decaytime, const float sustainlevel, const float releasetime, float* prevamplitude, float* stage) {
	// The amplitude variable
	float amplitude;
	// Compute the correct amplitude from the stage information
	switch ((int) *stage) {
		case 2: // Decay to the given sustain level
		case 3: // Or keep the sustain level
			amplitude = amplitudeenvelopedecay(decaytime, sustainlevel, prevamplitude);
			// If the amplitude has reached the sustain level, set the status to sustain
			if (*prevamplitude == 0.0) {
				*stage = AMPLITUDE_ENVELOPE_SUSTAIN;
			}
			// Otherwise keep the decay phase
			else {
				*stage = AMPLITUDE_ENVELOPE_DECAY;
			}
			break;
		case 4: // Decay (release) from the current level to zero level
			amplitude = amplitudeenveloperelease(releasetime, prevamplitude);
			// If the amplitude has reached the zero level, set the status to no sound
			if (*prevamplitude == 1.0) {
				*stage = AMPLITUDE_ENVELOPE_OFF;
			}
			// Otherwise keep the release phase
			else {
				*stage = AMPLITUDE_ENVELOPE_RELEASE;
			}
			break;
		case 5: // No output
			amplitude = 0.0;
			*stage = AMPLITUDE_ENVELOPE_OFF;
			break;
		default: // Attack
			amplitude = amplitudeenvelopeattack(attacktime, prevamplitude);
			// If the amplitude has reached the maximum, set the status to decay
			if (*prevamplitude >= 1) {
				*stage = AMPLITUDE_ENVELOPE_DECAY;
			}
			// Otherwise, keep the attack phase
			else {
				*stage = AMPLITUDE_ENVELOPE_ATTACK;
			}
			break;
	}
	// Return the envelope value
	return amplitude;
}

#ifdef __cplusplus
}
#endif

#endif
