/**
 * Flext external code for Max/MSP and PureData
 * Hammond organ model presented in the DAFx-11 paper
 *
 * version 0.1
 *
 * Created by Jussi Pekonen
 **/

/* Required libraries */
#include <flext.h> // For building compatible externals for both Max/MSP and Pd
#include "dafx11hammondorgan.h" // For the organ synthesis model
#include <vector> // For the polyphony data
#include "midifreq.h" // For the conversion from MIDI to frequencies

/** The hammond class **/
class hammond: public flext_dsp {
	
	/* The Flext header */
    FLEXT_HEADER(hammond, flext_dsp)
    
	/* Public methods and variables of the class */
	public:
		// Constructor
		hammond(int argc, const t_atom *argv);
		// Destructor
		~hammond() {
			// Free the sine table
			additivesynthesizer_tablefree(sinetable);
		}
	
	/* Protected methods and variables of the class */
	protected:
		// DSP calculation
		virtual void m_signal(int n, float *const *in, float *const *out);
	
	/* Private methods and variables of the class */
	private:
		// The sine table size
		int sinetablesize;
		// The sine table pointer
		float* sinetable;
		// The master phase
		float masterphase;
		// The master phase step size
		float masterphasestepsize;
		// The amplitude threshold
		float ampthreshold;
		// The frequency limit
		float maxfreq;
		// Polyphony number
		int hammondpolyphony;
		// Polyphony data vector
		std::vector<std::vector<float> > hammondpolyphonydata;
		// Drawbar data
		float hammonddrawbar[9];
		// Click toggle
		char hammondclick;
		// Harmonic imperfection toggle
		char hammondharmimperfection;
		// Disk imperfection toggle
		char hammonddiskimperfection;
		// Disk imperfection depth
		float hammonddiskimpdepth;
		// The MIDI processing function
		FLEXT_CALLBACK_V(processMIDI);
		void processMIDI(int argc, const t_atom *argv);
		// The drawbar data processing function
		FLEXT_CALLBACK_V(processDrawbar);
		void processDrawbar(int argc, const t_atom *argv);
		// The key click toggle function
		FLEXT_CALLBACK_1(setClick, float);
		void setClick(float click);
		// The harmonic imperfection toggle function
		FLEXT_CALLBACK_1(setHarmImperfection, float);
		void setHarmImperfection(float imperfection);
		// The disk imperfection toggle function
		FLEXT_CALLBACK_1(setDiskImperfection, float);
		void setDiskImperfection(float imperfection);
	
};

/** Associate the external to an object name **/
FLEXT_NEW_DSP_V("hammond~ dafx11organ~", hammond)

/** Implementation of the class constructor **/
hammond::hammond(int argc, const t_atom *argv) {
	
	/* Check the creation parameters */
	sinetablesize = 64;
	hammondpolyphony = 10;
	hammonddiskimpdepth = 0.05;
	switch (argc) {
		case 3:
			hammonddiskimpdepth = IsFloat(argv[2]) ? GetFloat(argv[2]) : IsInt(argv[2]) ? (float) GetInt(argv[2]) : 0.05;
		case 2:
			sinetablesize = IsFloat(argv[1]) ? (int) GetFloat(argv[1]) : IsInt(argv[1]) ? GetInt(argv[1]) : 64;
		case 1:
			hammondpolyphony = IsFloat(argv[1]) ? (int) GetFloat(argv[1]) : IsInt(argv[1]) ? GetInt(argv[1]) : 10;
			break;
		default:
			sinetablesize = 64;
			hammondpolyphony = 10;
			hammonddiskimpdepth = 0.05;
	}
	
	/* Initialize the Hammond parameters */
	// Set the initial values of the drawbars
	hammonddrawbar[0] = 0.0;
	hammonddrawbar[1] = 0.0;
	hammonddrawbar[2] = 1.0;
	hammonddrawbar[3] = 0.2;
	hammonddrawbar[4] = 0.2;
	hammonddrawbar[5] = 0.0;
	hammonddrawbar[6] = 0.0;
	hammonddrawbar[7] = 0.0;
	hammonddrawbar[8] = 0.1;
	// Set the click mode off
	hammondclick = 0;
	// Set the harmonic component imperfection mode off
	hammondharmimperfection = 0;
	// Set the disk imperfection mode off
	hammonddiskimperfection = 0;
		
	/* Initialize the additive synthesizer */
	// Set the look-up table
	sinetable = additivesynthesizer_tableinit(sinetablesize);
	// Set the master phase
	masterphase = 0;
	// Set the master oscillator frequency
	masterphasestepsize = 5.0;
	// Set the minimum amplitude threshold of the synthesis engine
	ampthreshold = 0.0;
	// Set the maximum frequency of the synthesis engine
	maxfreq = 10000.0;
	
	/* Inlets and outlets */
	// Inlets
	AddInAnything("Notes to be played as a list of a MIDI note data pair (note + velocity)");
	AddInAnything("The drawbar controls as a list (floats in range from 0 to 8)");
	AddInFloat("Key click toggle (0 = off, anything else = on)");
	AddInFloat("Harmonic imperfection toggle (0 = off, anything else = on)");
	AddInFloat("Disk imperfection toggle (0 = off, anything else = on)");
	// Outlets
	AddOutSignal("The organ tone");
	AddOutSignal("The master clock sinusoid (for tremolo)");
	
	/* Assign functions for inlets 1-5 */
	FLEXT_ADDMETHOD(0, processMIDI);
	FLEXT_ADDMETHOD(1, processDrawbar);
	FLEXT_ADDMETHOD(2, setClick);
	FLEXT_ADDMETHOD(3, setHarmImperfection);
	FLEXT_ADDMETHOD(4, setDiskImperfection);

}

/** Implementation of the processMIDI function **/
void hammond::processMIDI(int argc, const t_atom *argv) {
	// Variables for the MIDI data handling
	float midinumber, velocity;
	
	/* MIDI data checking */
	// There is a MIDI note and velocity pair (both are floats or integers)
	if ((argc == 2) && ((IsFloat(argv[0]) && IsFloat(argv[1])) || (IsInt(argv[0]) && IsInt(argv[1])))) {
		// Set the MIDI note
		midinumber = (IsFloat(argv[0])) ? GetFloat(argv[0]) : (float) GetInt(argv[0]);
		// Set the velocity
		velocity = (IsFloat(argv[1])) ? GetFloat(argv[1]) : (float) GetInt(argv[1]);
	}
	// The data is not valid
	else {
		return;
	}
	
	/* Variables for the MIDI processing */
	// Vector for the note data
	std::vector<float> notedata, newnotedata;
	// Temporary variables
	int k, found = -1;
	float freq;
	
	/* Check is the given note already in the note data vector */
	for (k = 0; ((k < hammondpolyphonydata.size()) && (found == -1)); k++) {
		// The data vector
		notedata = hammondpolyphonydata[k];
		// The note is already there
		if (notedata[0] == midinumber) {
			// Set the found flag on
			found = k;
		}
	}
	
	/* Maximum polyphony in use, remove the most oldest one, if the given note is not there */
	if ((hammondpolyphonydata.size() == hammondpolyphony) && (found == -1)) {
		// Step through the note data vector
		for (k = 0; k < hammondpolyphonydata.size(); k++) {
			// The data vector
			notedata = hammondpolyphonydata[k];
			// The oldest note that is in the release phase
			if (notedata[2] == AMPLITUDE_ENVELOPE_RELEASE) {
				// Remove it
				hammondpolyphonydata.erase(hammondpolyphonydata.begin() + k);
				// Break from the loop
				break;
			}
		}
		// If there is no notes in the release phase, remove the oldest
		if (hammondpolyphonydata.size() == hammondpolyphony) {
			// Remove the oldest note
			hammondpolyphonydata.erase(hammondpolyphonydata.begin());
		}
	}
	
	/* Process the given MIDI data */
	// The given note is already there
	if (found > -1) {
		// The data vector
		notedata = hammondpolyphonydata[found];
		// The note is re-excited
		if (velocity > 0) {
			// Set the amplitude envelope stage to attack
			notedata[2] = AMPLITUDE_ENVELOPE_ATTACK;
			// Set the amplitude envelope stage of the click to attack
			notedata[4] = AMPLITUDE_ENVELOPE_ATTACK;
			// Reset the data
			hammondpolyphonydata[found] = notedata;
		}
		// The note is released
		else {
			// Set the amplitude envelope stage to attack
			notedata[2] = AMPLITUDE_ENVELOPE_RELEASE;
			// Set the amplitude envelope stage of the click to attack
			notedata[4] = AMPLITUDE_ENVELOPE_RELEASE;
			// Reset the data
			hammondpolyphonydata[found] = notedata;
		}
	}
	// The given note is not in the note data vector, and it has been excited
	else if (velocity > 0) {
		// Set the note data
		newnotedata.push_back(midinumber);
		newnotedata.push_back(0);
		newnotedata.push_back(AMPLITUDE_ENVELOPE_ATTACK);
		newnotedata.push_back(0);
		newnotedata.push_back(AMPLITUDE_ENVELOPE_ATTACK);
		newnotedata.push_back(0);
		if (hammondharmimperfection) {
			freq = midifreq(midinumber - 12);
			newnotedata.push_back(freq);
			freq = midifreq(midinumber + 7);
			newnotedata.push_back(freq);
			freq = midifreq(midinumber);
			newnotedata.push_back(freq);
			freq = midifreq(midinumber + 12);
			newnotedata.push_back(freq);
			freq = midifreq(midinumber + 19);
			newnotedata.push_back(freq);
			freq = midifreq(midinumber + 24);
			newnotedata.push_back(freq);
			freq = midifreq(midinumber + 28);
			newnotedata.push_back(freq);
			freq = midifreq(midinumber + 31);
			newnotedata.push_back(freq);
			freq = midifreq(midinumber + 36);
			newnotedata.push_back(freq);
			newnotedata.push_back(midinumber - 12);
			newnotedata.push_back(midinumber + 7);
			newnotedata.push_back(midinumber);
			newnotedata.push_back(midinumber + 12);
			newnotedata.push_back(midinumber + 19);
			newnotedata.push_back(midinumber + 24);
			newnotedata.push_back(midinumber + 28);
			newnotedata.push_back(midinumber + 31);
			newnotedata.push_back(midinumber + 36);
		}
		else {
			freq = midifreq(midinumber);
			newnotedata.push_back(0.5*freq);
			newnotedata.push_back(1.5*freq);
			newnotedata.push_back(freq);
			newnotedata.push_back(2.0*freq);
			newnotedata.push_back(3.0*freq);
			newnotedata.push_back(4.0*freq);
			newnotedata.push_back(5.0*freq);
			newnotedata.push_back(6.0*freq);
			newnotedata.push_back(8.0*freq);			
			newnotedata.push_back(freqmidi(0.5*freq));
			newnotedata.push_back(freqmidi(1.5*freq));
			newnotedata.push_back(midinumber);
			newnotedata.push_back(midinumber + 12);
			newnotedata.push_back(freqmidi(3.0*freq));
			newnotedata.push_back(midinumber + 24);
			newnotedata.push_back(freqmidi(5.0*freq));
			newnotedata.push_back(freqmidi(6.0*freq));
			newnotedata.push_back(midinumber + 36);
		}
		// Push the note data to the note data vector
		hammondpolyphonydata.push_back(newnotedata);
	}

}

/** Implementation of the processMIDI function **/
void hammond::processDrawbar(int argc, const t_atom *argv) {
	/* Input data check */
	// Check that all input is floating point numbers
	int k;
	for (k = 0; k < argc; k++) {
		// If the data is not a float, return
		if (!IsFloat(argv[k])) {
			return;
		}
	}
	
	/* Process data */
	for (k = 0; k < 9 && k < argc; k++) {
		hammonddrawbar[k] = (GetFloat(argv[k])/8.0 > 1) ? 1.0 : (GetFloat(argv[k]) < 0) ? 0.0 : GetFloat(argv[k])/8.0;
	}

}

/** Implementation of the setClick function **/
void hammond::setClick(float click) {
	// Set the click toggle
	hammondclick = (char) click;
}

/** Implementation of the setHarmImperfection function **/
void hammond::setHarmImperfection(float imperfection) {
	// Set the harmonic imperfection toggle
	hammondharmimperfection = (char) imperfection;
}

/** Implementation of the setDiskImperfection function **/
void hammond::setDiskImperfection(float imperfection) {
	// Set the harmonic imperfection toggle
	hammonddiskimperfection = (char) imperfection;
}

/** Implementation of the virtual DSP function **/
void hammond::m_signal(int n, float *const *in, float *const *out) {
	/* Variables */
	// The output signal outlet pointer
	float* signal = out[0];
	// The tremolo signal outlet pointer
	float* tremolo = out[1];
	// Internal variable for storing the output signal value
	float soutput;
	// The sample rate, required for the phase stepping
	float Fs = Samplerate();

	// Note data entry
	std::vector<float> notedata;
	// The frequencies of the components
	float freq[9];
	// The phases of the components
	float phases[9];
	// The amplitudes of the components
	float amplitudes[9];
	// Stepping variables
	int k, l;
	// The general amplitude envelope parameters
	float amplitude;
	// The key click amplitude envelope parameters
	float camp, cstage = 1;
	
	/* Compute the output */
	// Generate the required number of samples
	while (n--) {
		// Compute the polyphonic components
		soutput = 0.0;
		// Step through each note
		for (k = 0; k < hammondpolyphonydata.size(); k++) {
			// The note data
			notedata = hammondpolyphonydata[k];
			// Set the component frequencies phases, and amplitudes
			for (l = 0; l < 9; l++) {
				freq[l] = notedata[l+6];
				phases[l] = freq[l]/freq[0]*notedata[1];
				// Model the disk imperfection
				camp = 0.0;
				if (hammonddiskimperfection) {
					camp = additivesynthesizer_tablelookup(masterphase + notedata[15 + l]/sinetablesize, sinetable, sinetablesize);
				}
				amplitudes[l] = (1.0 + camp*hammonddiskimpdepth)*hammonddrawbar[l];
			}
			// Compute the amplitude envelope value
			amplitude = amplitudeenvelope(0.005*Fs, 0.05*Fs, 0.99, 0.005*Fs, &notedata[3], &notedata[2]);
			// Compute the sample for that note
			soutput += amplitude*additivesynthesizer(sinetable, sinetablesize, 9, freq, phases, amplitudes, Fs, ampthreshold, maxfreq);
			// Update the phase data
			notedata[1] = (phases[0] > 1.0) ? (phases[0] - 1.0) : phases[0];
			// Compute the key click, if needed
			if (hammondclick) {
				// Compute the click amplitude envelope value
				camp = amplitudeenvelope(0.005*Fs, 0.07*Fs, 0, 0.005*Fs, &notedata[5], &notedata[4]);
				// Compute the click
				soutput += 0.2*camp*additivesynthesizer(sinetable, sinetablesize, 1, &freq[7], &phases[7], &cstage, Fs, ampthreshold, maxfreq);
			}
			// Update the note data vector accordingly
			hammondpolyphonydata[k] = notedata;
			// The sound has died out, remove from the note data vector
			if (notedata[2] == AMPLITUDE_ENVELOPE_OFF) {
				// Remove
				hammondpolyphonydata.erase(hammondpolyphonydata.begin() + k);
				k--;
			}
		}

		// Output the sample
		*signal++ = soutput;
		// Output the tremolo oscillator
		*tremolo++ = additivesynthesizer(sinetable, sinetablesize, 1, &masterphasestepsize, &masterphase, &cstage, Fs, ampthreshold, maxfreq);
		// Update the master phase
		masterphase = (masterphase > 1.0) ? (masterphase - 1.0) : (masterphase);
	}
}
