#
# Vector Phase Distortion - formants
#
from numpy import pi,cos
from scipy.signal import *
from pylab import *
from matplotlib.font_manager import FontProperties

f0 = 500.
fs = 44100.
w0 = 2*pi*f0/fs
N = int(fs/f0)
t = arange(0,44100)
T = f0/fs
P = 1/T

# -- phase distortion function
def vpd(x,d,v):
	if x < d:		return (v*x)/d
	else:				return (1-v)*(x-d)/(1-d) + v

# -- signal generation
def process(d,v):
	y = zeros(len(t))
	x = zeros(len(t))
	phi = 0
	for n in range(0,len(t)):
		x[n] = vpd(phi,d,v)		# vector phase distortion
		phi += T
		if phi > 1: phi -= 1
	y = -cos(2*pi*x)				# waveshaping
	return y,x

# -- spectrum
def spectrum(s):
	NFFT = 16384*2
	NWIN = 16384
	win = chebwin(NWIN, 120)
	win = append(win, zeros(NFFT-NWIN))
	scal = NFFT*sqrt(mean(win**2))
	spec = fft(win*s[0:NFFT])
	mags = sqrt(spec[0:NFFT/2].real**2 + spec[0:NFFT/2].imag**2)
	norm = 20*log10(mags/scal)
	spec = norm - max(norm)
	return spec


# -- main
y1,x1 = process(0.5, 3)
spec1 = spectrum(y1)
y2,x2 = process(0.3, 3)
spec2 = spectrum(y2)


# -- plotting
fig = figure(facecolor='#e2e2e2', figsize=(10,6))
fontcolor = '#000000'
font = FontProperties(family='serif', size=18)

p1 = subplot(221)
p1.plot(t,y1, 'k', lw=1)
p1.plot(t,x1, 'k--', lw=1)
grid(True)
xlim(0,2.1*N)
ylim(-1.1,3.1)

labels = p1.get_xticklabels() + p1.get_yticklabels()
setp(labels, color=fontcolor, fontproperties=font)
p1.xaxis.set_ticks([0,0.5*P,P,1.5*P,2*P])
p1.xaxis.set_ticklabels(["0","0.5","1","1.5","2"])
xlabel('Time (periods)', color=fontcolor, fontproperties=font)
ylabel('Level', color=fontcolor, fontproperties=font)

p2 = subplot(222)
p2.plot(linspace(0,fs/2,len(spec1)), spec1, 'k', lw=1)
grid(True)
xlim(0,fs/2)
ylim(-80,6)

labels = p2.get_xticklabels() + p2.get_yticklabels()
setp(labels, color=fontcolor, fontproperties=font)
p2.xaxis.set_ticklabels(["0","5","10","15","20"])
p2.yaxis.set_ticks([-80,-60,-40,-20,0])
xlabel('Frequency (kHz)', color=fontcolor, fontproperties=font)
ylabel('Magnitude (dB)', color=fontcolor, fontproperties=font)

p3 = subplot(223)
p3.plot(t,y2, 'k', lw=1)
p3.plot(t,x2, 'k--', lw=1)
grid(True)
xlim(0,2.1*N)
ylim(-1.1,3.1)

labels = p3.get_xticklabels() + p3.get_yticklabels()
setp(labels, color=fontcolor, fontproperties=font)
p3.xaxis.set_ticks([0,0.5*P,P,1.5*P,2*P])
p3.xaxis.set_ticklabels(["0","0.5","1","1.5","2"])
xlabel('Time (periods)', color=fontcolor, fontproperties=font)
ylabel('Level', color=fontcolor, fontproperties=font)

p4 = subplot(224)
p4.plot(linspace(0,fs/2,len(spec1)), spec2, 'k', lw=1)
grid(True)
xlim(0,fs/2)
ylim(-80,6)

labels = p4.get_xticklabels() + p4.get_yticklabels()
setp(labels, color=fontcolor, fontproperties=font)
p4.xaxis.set_ticklabels(["0","5","10","15","20"])
p4.yaxis.set_ticks([-80,-60,-40,-20,0])
xlabel('Frequency (kHz)', color=fontcolor, fontproperties=font)
ylabel('Magnitude (dB)', color=fontcolor, fontproperties=font)

# fig.subplots_adjust(hspace=0.28)
fig.subplots_adjust(left=0.1,bottom=0.1,right=0.96,top=0.96,wspace=0.29,hspace=0.39)

show()
