function out = vintage(in,type,a,wn,rn,play)
% Vintage telephone sound effect according to S. Oksanen and V. Vlimki,
% "Digital Modeling of the Vintage Telephone Sound", 
% in Proc. ICMC2011, Huddersfield, UK, Jul. 31 - Aug. 5, 2011
%
% Usage: 
% out = vintage(in,type,a,wn,rn,play) 
%
% in    = input data vector, fs 44.1 kHz
% type  = filter type, (1,2)
% a     = nonlinearity coefficient, [0...1]
% wn    = white noise variance, [0...1]
% wr    = rugged noise variance, [0...1]
% play  = sound output, (1 = sound, 0 = no sound)
% out   = output data vector
%
% Example use case:
% out = vintage(in,1,0.2,0.1,0.1,1)
%
% Required functions:
% - param_peak_boost.m, written by Henri Penttinen
% - envel.m, written by Matti Karjalainen
%
% Author: Sami Oksanen
% Last modified: Aug. 10, 2011
% Version: 1.0, first release
%
% License info:
% This work is licensed under a Creative Commons
% Attribution-NonCommercial-ShareAlike 3.0 Unported License.
%% Static variables
fs = 44100; % sample rate, fixed at the moment
plots = 0; % plot switch for param_peak_boost

%EQ filter parameter selection, fc, G, Q, for details see Table 1 on the paper 
if(type==1)
    param = [1800 4300; 22 6; 4 8]; % Type I
else
    param = [280 2000 3900; 8 15 21; 0.9 3 2.5]; % Type II
end

%% Pre-filter, Sec. 3.1
out = in; % output vector allocation

for i=1:size(param,2)
    f_c = param(1,i); % EQ center frequency
    G = param(2,i);   % EQ gain
    Q = param(3,i);   % EQ Q-value
    [B,A] = param_peak_boost(f_c,G,Q,fs,plots);
    out = filter(B,A,out);
end

out = 1.0 * out./max(abs(out)); % output scaling 

%% Nonlinearity, Sec 3.2
if(a>0)
    u = out;
    nonl = a^4*u.^5 + a^4*u.^4 - a^3*u.^4 - a^3*u.^3 + a^2*u.^3 + a^2*u.^2 - a*u.^2 - a*u + u; % static nonlinearity Eq. 14 
    nonl(nonl>1) = 1;   % clipping
    nonl(nonl<-1) = -1; % clipping
    out = nonl;
else
    out = out./max(abs(out)); % output scaling
end
%% Noise addition, Sec 3.3
noise = (2*rand(length(in),1)-1); % white noise

en = envel(in,700); % envelope follower
[bb,aa] = butter(4,[1000/(fs/2) 4800/(fs/2)]); % BPF for noise
bp_noise = en.*filter(bb,aa,2*rand(length(in),1)-1); % BP-filtered noise
bp_noise = bp_noise./max(abs(bp_noise));  % normalization

out = out + wn*noise + rn*(bp_noise);

%% Post-filter, Sec. 3.4
[B,A] = butter(4,[400/(fs/2) 2800/(fs/2)]); % bandpass filter
out = filter(B,A,out);
out = out./max(abs(out)); % output scaling

%% sound output
if(play); soundsc(out,fs); end;
