% ODOMP.m - Orthogonal Matching Pursuit for Over-Determined dictionary Matrix
%
% ODOMP returns the vector y that minimizes sum(abs(y)>0)
%   subject to norm(By-t,inf) <= xi.
% Usage: [y, feasible, err] = ODOMP(N, B, t, thr, W)
%
% Input:
% N = maximum number of iteration
% B = Over-Determined dictionary Matrix
% t = signal
% thr = threshold for reconstruction error
%
% W(optional) = weigths for reconstruction error
%
% Output:
% y = dictionary coefficient (the sparse solution)
% feasible(boolean) = the solution y is feasible or not
% err = error for each iteration
%
% Created by Mario Antonelli, August 20, 2021



function [y, feasible, err] = ODOMP(N, B, t, thr, W)

warning('off', 'all')
if nargin<4
    W = ones(size(t));
end
% N:number of iterations, B: dictionary matrix, t: signal
B0=B;
n=size(B);
B1=zeros(n);
R=t;
if(N <= 0)
    error('The number of iterations needs to be greater then 0')
end
err = nan(N,1);
feasible = false;
for k=1:N
    [~,d] = max(abs(B'*R));
    B1(:,d)=B(:,d);
    B(:,d)=0;
    y = B1\t;
    R = t - B1*y;
    err(k) = norm(W*(B0*y-t), 'inf');
    if err(k)<thr
        feasible = true;
        break
    end
end
warning('on', 'all')