function fig07_fbam_stabalias()
% fig 7 -- FBAM stability and aliasing
% (this takes about 42 minutes to compute on a 2 GHz laptop)
%
% Feedback Amplitude Modulation Synthesis
% J.Kleimola, V.Lazzarini, V.Vlimki, J.Timoney
% 2010
%
    fs = 44100;
    t = linspace(0,44100,44100);
    NFFT = 16384;

    % -- cover 88-key piano keyboard
    f1 = 27.5;
    f2 = 4186.01;
    P1 = ceil(fs/f1);
    P2 = floor(fs/f2);

    % ---------------------------------------------------------------------
    % stability
    % returns beta that keeps the system stable
    %
    function B = beta_stability(N)
        gN = 1;
        for n = 1:N
            gN = gN * cos(2*pi*n/N);
        end
        % gN * B**(N-1) = 1
        B = power(1./abs(gN), 1./(N-1));
    end

    % ---------------------------------------------------------------------
    % aliasing
    % returns beta that keeps the aliasing below -80 dB
    %
    function B = beta_aliasing(f0)
        dB = 0.01;
        % -- some defaults to boost computation
        if   f0 < 600,      B = 1.5;
        elseif f0 < 1500,   B = 1.0;
        elseif f0 < 2300,   B = 0.7;
        elseif f0 < 3140,   B = 0.5;
        else                B = 0.2;
        end
        %
        w0 = 2*pi*f0/fs;
        x = cos(w0*t);
        %
        h = floor((fs/2)/f0) + 1;   % first aliasing harmonic
        fa = fs-h*f0;               % highest folding frequency
        bin = floor(fa*NFFT/fs);    % freq = bin*fs/NFFT
        mag = -1000;
        while mag < -80
            B = B + dB;
            % -- fbam
            y = zeros(1,length(t));
            y(1) = 0;
            for n = 2:length(t)
                y(n) = x(n) * (1 + B*y(n-1));
            end
            %
            spec = fbam_magspect(y/max(y),16384);
            mag = max(spec(bin-2:bin+2));
            if isinf(mag)           % numerical range may run out for very low frequencies
                B = 2+dB;           % (they do not alias). Use maximum beta
                break
            end
        end
        B = B-dB;
    end
    
    tic

    % -- stability
    sp = P1-2:-4:0;
    sf = linspace(f1,f2,length(sp));
    sb = zeros(length(sf),1);
    si = 1;
    for p = P1-2:-4:0
        sb(si) = beta_stability(p);
        sf(si) = fs/p;
        si = si + 1;
    end
    
    % -- aliasing
    NF = 100;
    bf = linspace(f1,f2,NF);
    bb = zeros(NFFT/2,1);
    bi = 1;
    for i = 1:length(bf)
        f = bf(i);
        bb(bi) = beta_aliasing(f);
        bi = bi + 1;
    end

    % -- aliasing (2x oversampling)
    fs = 2*fs
    b2 = zeros(NFFT/2,1);
    bi = 1;
    for i = 1:length(bf)
        f = bf(i);
        b2(bi) = beta_aliasing(f);
        bi = bi + 1;
    end
    
    toc
    
    % -- plotting
    figure('Position',[100 100 800 400])
    axes('Position', [0.1 0.2 0.8 0.75])
    plot(bf, bb(1:length(bf)), 'k', 'LineWidth', 1);
    % title('Fig 7 -- FBAM stability and aliasing')
    hold on;
    plot(bf, b2(1:length(bf)), 'k.', 'LineWidth', 1);
    plot(sf, sb(1:length(sf)), 'k--', 'LineWidth', 1);
    grid on;
    xlim([f1,f2]);
    ylim([0,2.05]);
    
    set(gcf, 'Color', [0.9,0.9,0.9])
    set(gca, 'YTick', [0:0.2:2.01]);
    set(gca ,'FontName', 'times')
    set(gca, 'FontSize', 14)
    xlabel('Frequency (Hz, 88-key piano range)');
    ylabel('\it\beta');

end