function fig23_fbam_closedform()
% fig 23 -- FBAM closed form approximation
% Feedback Amplitude Modulation Synthesis
% J.Kleimola, V.Lazzarini, V.Vlimki, J.Timoney
% 2010
%
    fs = 44100;
    f0 = 441;
    w0 = 2*pi*f0/fs;
    N = floor(fs/f0);
    L = 500*N;
    dur = L+N;
    NFFT = 16384;    
    t = 1:dur;

    
    % -- FBAM ------------------------------------------------------------
    a = cos(w0*t);              % time variant coefficients
    x = a;                      % input
    beta = 1.0;
    fbam = zeros(1,length(t));
    fbam(1) = 0;
    for n = 2:length(t)
        fbam(n) = x(n) + beta * a(n) * fbam(n-1);
    end
    fbam = fbam / max(abs(fbam));

    % -- Jolley (593), sin of the numerator changed to cos ---------------
    wc = w0;        % theta
    wm = 1*wc;      % beta
    b = 2.5;        % b**2 > 1 (or b=0.40 for b**2 < 1)
    dsf = zeros(1,length(t));
    for n = 1:length(t)
        dsf(n) = cos(wc*t(n)) / (1 - 2*b*cos(wm*t(n)) + b*b)^3;
    end    
    dsf = dsf / max(dsf);

    % -- spectra and spectral envelopes ----------------------------------
    spec1 = fbam_magspect(fbam, NFFT);
    spec2 = fbam_magspect(dsf, NFFT);
    nharms = floor((fs/2)/f0) + 1;
    sf = zeros(nharms-1);
    sm1 = zeros(nharms-1);
    sm2 = zeros(nharms-1);
    f = 0;
    i = 1;
    while f < fs/2
        bin = round((NFFT)*f/fs) + 1;
        sf(i) = 2*f/fs;
        sm1(i) = spec1(bin);
        sm2(i) = spec2(bin);
        f = f + f0;
        i = i + 1;
    end

    % -- waveshaper ------------------------------------------------------
    phi = 0:f0/fs:1+f0/fs;
    p = 2*phi - 1;
    g = zeros(1,length(p));
    for n = 1:length(p)
        g(n) = p(n) / (1 - 2*b*p(n) + b*b)^3;
    end
    g = g / max(g);
    ws = spline(p,g);
    sws = ppval(ws, x);
    sws = sws / max(sws);

    
    % -- waveform plots
    figure('Position', [50 200 1100 280])
    axes('Position', [0.1 0.25 0.2 0.7])
    L = 2*N;
    fbam_plotsig(t, dsf, L);
    hold on;
    plot(t(1:L),fbam(6:L+5), 'k--', 'LineWidth', 1);
    ylim([-0.2,1.1]);
    set(gca, 'YTick', [-0.2:0.2:1.0] )
    set(gca, 'XTick', [0:50:200] )
    
    % -- spectrum plots
    axes('Position', [0.42 0.25 0.2 0.7])
    f = 0:2/NFFT:1-2/NFFT;
    plot(f, spec2, 'k', 'LineWidth', 1);
    hold on;
    plot(sf,sm1, 'k--', 'LineWidth', 1);
    axis([0 0.5 -100 5])
    grid on
    set(gca ,'FontName', 'times')
    set(gca, 'FontSize', 14)
    xlabel('Frequency (kHz)')
    ylabel('Magnitude (dB)')
    set(gca, 'YTick', [-100:20:0] )
    set(gca, 'XTick', [0:0.5/5:0.5] )
    set(gca, 'XTickLabel', {'0','2','4','6','8','10'} )

    % -- waveshaper plot
    axes('Position', [0.74 0.25 0.2 0.7])
    plot(p,g,'k', 'LineWidth', 1);
    hold on;
    plot(cos(2*pi*phi), p, 'Color', [0.7 0.7 0.7]);
    plot(p, sws(1:length(p)), 'k--', 'LineWidth', 1);
    grid on;
    xlim([-1.1,1.1]);
    ylim([-1.1,1.1]);
    set(gca, 'YTick', [-1:0.5:1] )
    set(gca, 'XTick', [-1:0.5:1] )
    set(gca ,'FontName', 'times');
    set(gca, 'FontSize', 14);
    xlabel('Input');
    ylabel('Output');
end