// ----------------------------------------------------------------------------
// fbam~ PD external
// Jari Kleimola 2010
//
#define __attribute__(x)	;
#include "m_pd.h"
#include <math.h>

#define c_maxDelay	4096

static t_class* fbam_class;
typedef struct _fbam
{
	t_object	x_obj;
	t_float	x_x;
	t_int		m_variation;			// variation number
	t_float	m_beta;					// beta
	t_int		m_ws;						// waveshaper
	t_int		m_D;						// variable delay length (in samples)
	t_int		m_id;						// index for variable delay var 3.5
	t_sample	m_delay[c_maxDelay];	// delay buffer for var 3.5
	t_float	m_y1,m_x1;				// output and input delays
} t_fbam;

// ----------------------------------------------------------------------------
//
static t_floatarg limit(t_floatarg var, t_floatarg min, t_floatarg max)
{
	if (var < min)			var = min;
	else if (var > max)	var = max;
	return var;
}

// ----------------------------------------------------------------------------
//
static t_float getSample(t_fbam* fbam)
{
	return fbam->m_delay[fbam->m_id];
}

// ----------------------------------------------------------------------------
//
static void putSample(t_fbam* fbam, t_float s)
{
	fbam->m_delay[fbam->m_id] = s;
	fbam->m_id++;
	if (fbam->m_id >= fbam->m_D)
		fbam->m_id = 0;
}

// ----------------------------------------------------------------------------
// ctor
//
static void* fbam_new()
{
	t_fbam* fbam = (t_fbam*)pd_new(fbam_class);

	fbam->x_x = 0;
	fbam->m_variation = 0;
	fbam->m_beta = 1;
	fbam->m_D = 1;
	fbam->m_id = 0;
	fbam->m_y1 = 0;
	fbam->m_x1 = 0;

	inlet_new(&fbam->x_obj, &fbam->x_obj.ob_pd, &s_signal, &s_signal);
	inlet_new(&fbam->x_obj, &fbam->x_obj.ob_pd, &s_signal, &s_signal);
	outlet_new(&fbam->x_obj, &s_signal);
	inlet_new(&fbam->x_obj, &fbam->x_obj.ob_pd, &s_float, gensym("v"));
	inlet_new(&fbam->x_obj, &fbam->x_obj.ob_pd, &s_float, gensym("beta"));
	inlet_new(&fbam->x_obj, &fbam->x_obj.ob_pd, &s_float, gensym("ws"));
	inlet_new(&fbam->x_obj, &fbam->x_obj.ob_pd, &s_float, gensym("D"));

	return fbam;
}

// ----------------------------------------------------------------------------
// inlets
//
static void fbam_set_var(t_fbam* fbam, t_floatarg v)		{ fbam->m_variation = limit(v, 0, 6); }
static void fbam_set_beta(t_fbam* fbam, t_floatarg beta)	{ fbam->m_beta = beta; }
static void fbam_set_ws(t_fbam* fbam, t_floatarg ws)		{ fbam->m_ws = limit(ws, 0, 1); }
static void fbam_set_D(t_fbam* fbam, t_floatarg D)			{ fbam->m_D = D; }


// ============================================================================
// DSP
// ============================================================================

// ----------------------------------------------------------------------------
// the basic feedback AM equation (25), inputs decoupled as in eq (44)
//
static void processBasic(t_int* p)
{
	t_fbam* fbam = (t_fbam*)p[1];
	t_sample* car = (t_float*)p[2];
	t_sample* mod = (t_float*)p[3];
	t_sample* out = (t_float*)p[5];
	int n = (int)p[6];
	int i;
	t_float beta = fbam->m_beta;
	t_float y1 = fbam->m_y1;

	for (i=0; i<n; i++)
	{
		t_float x = *car++;
		t_float m = *mod++;
		y1 = x + beta*m*y1;
		*out++ = y1;
	}

	fbam->m_y1 = y1;
}

// ----------------------------------------------------------------------------
// var1: feedforward delay variation eq (27), inputs decoupled as in eq (44)
//
static void process31(t_int* p)
{
	t_fbam* fbam = (t_fbam*)p[1];
	t_sample* car = (t_float*)p[2];
	t_sample* mod = (t_float*)p[3];
	t_sample* out = (t_float*)p[5];
	int n = (int)p[6];
	int i;
	t_float beta = fbam->m_beta;
	t_float y1 = fbam->m_y1;
	t_float x1 = fbam->m_x1;

	for (i=0; i<n; i++)
	{
		t_float x = *car++;
		t_float m = *mod++;
		y1 = x1 - x - beta*m*y1;
		x1 = x;
		*out++ = y1;
	}

	fbam->m_y1 = y1;
	fbam->m_x1 = x1;
}

// ----------------------------------------------------------------------------
// var2: coeff-mod allpass eq (29), inputs decoupled as in eq (44)
//
static void process32(t_int* p)
{
	t_fbam* fbam = (t_fbam*)p[1];
	t_sample* car = (t_float*)p[2];
	t_sample* mod = (t_float*)p[3];
	t_sample* out = (t_float*)p[5];
	int n = (int)p[6];
	int i;
	t_float beta = fbam->m_beta;
	t_float y1 = fbam->m_y1;
	t_float x1 = fbam->m_x1;

	for (i=0; i<n; i++)
	{
		t_float x = *car++;
		t_float m = *mod++;
		y1 = x1 - beta*m*x + beta*m*y1;
		x1 = x;
		*out++ = y1;
	}

	fbam->m_y1 = y1;
	fbam->m_x1 = x1;
}

// ----------------------------------------------------------------------------
// var3-I: heterodyning eq I (30), inputs decoupled as in eq (44)
//
static void process331(t_int* p)
{
	t_fbam* fbam = (t_fbam*)p[1];
	t_sample* car = (t_float*)p[2];
	t_sample* mod = (t_float*)p[3];
	t_sample* rm  = (t_float*)p[4];
	t_sample* out = (t_float*)p[5];
	int n = (int)p[6];
	int i;
	t_float beta = fbam->m_beta;
	t_float y1 = fbam->m_y1;

	for (i=0; i<n; i++)
	{
		t_float x = *car++;
		t_float m = *mod++;
		t_float r = *rm++;
		y1 = r * (x + beta*m*y1);
		*out++ = y1;
	}

	fbam->m_y1 = y1;
}

// ----------------------------------------------------------------------------
// var3-II: heterodyning eq II (31), inputs decoupled as in eq (44)
//
static void process332(t_int* p)
{
	t_fbam* fbam = (t_fbam*)p[1];
	t_sample* car = (t_float*)p[2];
	t_sample* mod = (t_float*)p[3];
	t_sample* rm  = (t_float*)p[4];
	t_sample* out = (t_float*)p[5];
	int n = (int)p[6];
	int i;
	t_float beta = fbam->m_beta;
	t_float y1 = fbam->m_y1;

	for (i=0; i<n; i++)
	{
		t_float x = *car++;
		t_float m = *mod++;
		t_float r = *rm++;
		y1 = x + beta*m*y1;
		*out++ = r * y1;
	}

	fbam->m_y1 = y1;
}

// ----------------------------------------------------------------------------
// var4: nonlinear waveshaping eq (35), inputs decoupled as in eq (44)
//
static void process34(t_int* p)
{
	t_fbam* fbam = (t_fbam*)p[1];
	t_sample* car = (t_float*)p[2];
	t_sample* mod = (t_float*)p[3];
	t_sample* out = (t_float*)p[5];
	int n = (int)p[6];
	int i;
	t_float beta = fbam->m_beta;
	t_float y1 = fbam->m_y1;

	if (fbam->m_ws == 0)
	{
		for (i=0; i<n; i++)
		{
			t_float x = *car++;
			t_float m = *mod++;
			y1 = x + m * cos(beta*y1);
			*out++ = y1;
		}
	}
	else
	{
		for (i=0; i<n; i++)
		{
			t_float x = *car++;
			t_float m = *mod++;
			y1 = x + m * fabs(beta*y1);
			*out++ = y1;
		}
	}

	fbam->m_y1 = y1;
}

// ----------------------------------------------------------------------------
// var5: non-unitary feedback periods eq (40), inputs decoupled as in eq (44)
//
static void process35(t_int* p)
{
	t_fbam* fbam = (t_fbam*)p[1];
	t_sample* car = (t_float*)p[2];
	t_sample* mod = (t_float*)p[3];
	t_sample* out = (t_float*)p[5];
	int n = (int)p[6];
	int i;
	t_float beta = fbam->m_beta;

	for (i=0; i<n; i++)
	{
		t_float x = *car++;
		t_float m = *mod++;
		t_float y = x + beta*m*getSample(fbam);
		putSample(fbam, y);
		*out++ = y;
	}
}

// ----------------------------------------------------------------------------
// main DSP processing method
//
static t_int* fbam_process(t_int* p)
{
	t_fbam* fbam = (t_fbam*)p[1];
	switch (fbam->m_variation)
	{
		case 0:	processBasic(p);	break;
		case 1:	process31(p);		break;
		case 2:	process32(p);		break;
		case 3:	process331(p);		break;
		case 4:	process332(p);		break;
		case 5:	process34(p);		break;
		case 6:	process35(p);		break;
	}

	// next DSP callback object (for PD)
	return (p+8);
}

// ----------------------------------------------------------------------------
// called when DSP is turned on. Registers the main DSP processing method.
//
static void fbam_dsp(t_fbam* fbam, t_signal** sp)
{
	dsp_add(fbam_process, 7, fbam, sp[0]->s_vec, sp[1]->s_vec, sp[2]->s_vec, sp[3]->s_vec, sp[0]->s_n);
}

// ----------------------------------------------------------------------------
// entry point 
//
void fbam_tilde_setup(void)
{
	fbam_class = class_new(gensym("fbam~"), (t_newmethod)fbam_new, (t_method)0, sizeof(t_fbam), CLASS_DEFAULT, 0);
	CLASS_MAINSIGNALIN(fbam_class, t_fbam, x_x);
	class_addmethod(fbam_class, (t_method)fbam_dsp, gensym("dsp"), 0);
	class_addmethod(fbam_class, (t_method)fbam_set_var, gensym("v"), A_FLOAT, 0);
	class_addmethod(fbam_class, (t_method)fbam_set_beta, gensym("beta"), A_FLOAT, 0);
	class_addmethod(fbam_class, (t_method)fbam_set_ws, gensym("ws"), A_FLOAT, 0);
	class_addmethod(fbam_class, (t_method)fbam_set_D, gensym("D"), A_FLOAT, 0);
}
