#
# FBAM scaling : GUI for testing
#
# Feedback Amplitude Modulation Synthesis
# J.Kleimola, V.Lazzarini, V.Vlimki, J.Timoney
# 2010
#
from scipy.signal import *
from pylab import *
from matplotlib.widgets import Slider
from matplotlib.font_manager import FontProperties

f0 = 441.
fs = 44100.
w = 2*pi*f0/fs
t = linspace(0,44100,44100)

B = 1          # beta
a = cos(w*t)   # coefficients
x = cos(w*t)   # input

# -- freq range is 88-key piano keyboard
f1 = 27.5
f2 = 4186.01


# -----------------------------------------------------------------------------
# straight FBAM theme
#
def fbam(f0,B):
   global a,x
   w = 2*pi*f0/fs
   a = cos(w*t)
   x = a
   y = zeros(len(a))
   y[0] = x[0]
   for n in range(1,len(a)):
      y[n] = x[n] + B*a[n]*y[n-1]
   return y

# -----------------------------------------------------------------------------
# normalized magnitude spectrum
#
NFFT = 16384 / 2
win = chebwin(NFFT, 120)
scal = NFFT*sqrt(mean(win**2))

def spectrum(sig):
   spec = fft(win*sig[0:NFFT])
   mags = sqrt(spec[0:NFFT/2].real**2 + spec[0:NFFT/2].imag**2)
   norm = 20*log10(mags/scal)
   norm = norm - max(norm)
   return norm

# -----------------------------------------------------------------------------
# scaling using polynomial/power approximations and linear interpolation
#
def scale_func(s,c,f0,B):
   g = 1
   # -- beta chooses the approximation
   blo = int(abs(B)/db)
   if blo >= nbeta:
      blo = nbeta-1
   bhi = blo + 1
   w_b = (B - blo*db) / db
   # -- linear interpolation
   if blo > 19:   g1 = c[blo][0] * pow(f0,c[blo][1])
   else:          g1 = polyval(c[blo], f0)
   if bhi > 19:   g2 = c[bhi][0] * pow(f0,c[bhi][1])
   else:          g2 = polyval(c[bhi], f0)
   g  = (1-w_b) * g1 + w_b * g2
   # -- scaling
   return s/g

# -----------------------------------------------------------------------------
# scaling using a 2D lookup table and bilinear interpolation
#
def scale_lookup(s,c,f0,B):
   # -- frequency
   flo = int((f0-f1)/dfreq)
   if flo >= nfreq:
      flo = nfreq-1
   fhi = flo + 1
   w_f = (f0 - (f1 + flo*dfreq)) / dfreq
   # -- beta
   blo = int(B/db)
   if blo >= nbeta:
      blo = nbeta-1
   bhi = blo + 1
   w_b = (B - blo*db) / db
   # -- bilinear interpolation
   g1 = (1-w_b) * c[flo,blo] + w_b * c[flo,bhi]
   g2 = (1-w_b) * c[fhi,blo] + w_b * c[fhi,bhi]
   g  = (1-w_f) * g1 + w_f * g2
   # -- scale
   return s/g


# -----------------------------------------------------------------------------
# load 2D lookup table
# values were computed using fbam_scaling.py
#
def scalingTable():
   c = genfromtxt("fbam_scale2D.txt")
   return c

# -----------------------------------------------------------------------------
# polynomial and power fits
# values were computed using fbam_scaling.py
#
def scalingCoeffs():
   coeffs = [
      # -- polynomial (beta = 0.00 - 0.95)
      # -- higher orders first, eg. c[0]*x^2 + c[1]*x + c[2]
      array([ -2.37551590e-19,  1.00000000e+00]),
      array([ -3.49626307e-06,  1.05364354e+00]),
      array([ -6.48755533e-06,  1.11298596e+00]),
      array([ -1.01977627e-05,  1.17936129e+00]),
      array([ -1.51877041e-05,  1.25418830e+00]),
      array([ -2.17339779e-05,  1.33910130e+00]),
      array([ -3.05601955e-05,  1.43633029e+00]),
      array([ -4.25216963e-05,  1.54874043e+00]),
      array([ -5.86920609e-05,  1.68001365e+00]),
      array([ -7.95525826e-05,  1.83484624e+00]),
      array([ -1.08016350e-04,  2.02056122e+00]),
      array([ -1.47181331e-04,  2.24673004e+00]),
      array([ -2.01970089e-04,  2.52778611e+00]),
      array([ -2.78955443e-04,  2.88540179e+00]),
      array([  2.75714854e-08, -4.81415790e-04,  3.38122006e+00]),   # 0.7
      array([  7.07697279e-08, -7.93581980e-04,  4.06413884e+00]),
      array([ -1.52839574e-11,  2.43749518e-07, -1.46575330e-03, 5.09157601e+00]),  # 0.8
      array([ -8.44794540e-11,  8.16289887e-07, -3.05286764e-03, 6.79758363e+00]),
      array([ -3.41038121e-17,  4.48577258e-13, -2.31453620e-09, 6.07798282e-06, -9.22305689e-03, 1.02628135e+01]),  # 0.9
      array([ -3.66063587e-16,  4.24995387e-12, -1.83647407e-08, 3.68263032e-05, -3.59415248e-02, 1.96008498e+01]),
      # -- power (beta = 1.00 - 1.95)
      # -- y = c[0] * x^c[1]
      # -- these do not work very well
      array([ 1105.4,  -0.725]),
      array([ 5260.8,  -0.916]),
      array([ 21211.8, -1.085]),
      array([ 304035,  -1.417]),
      array([ 6e+06,   -1.798]),
      array([ 2e+08,   -2.213]),
      array([ 6e+09,   -2.654]),
      array([ 2e+11,   -3.112]),
      array([ 1e+13,   -3.583]),
      array([ 4e+14,   -4.061]),
      array([ 2e+16,   -4.543]),
      array([ 1e+18,   -5.027]),
      array([ 5e+19,   -5.509]),
      array([ 2e+21,   -5.989]),
      array([ 9e+22,   -6.465]),
      array([ 4e+24,   -6.934]),
      array([ 2e+26,   -7.395]),
      array([ 6e+27,   -7.841]),
      array([ 2e+29,   -8.272])
      ]
   return coeffs

# -----------------------------------------------------------------------------
# 1 - 2D table lookup with bilinear interpolation
# 2 - polynomial and power approximations
#
def setScalingMode(mode):
   global scale,stab,nbeta
   if mode == 1:
      scale = scale_lookup
      stab = c1
   elif mode == 2:
      scale = scale_func
      stab = c2


# -- common scaling values
nfreq = 100                # 100 frequency samples (linear)
nbeta = 2*19               # 38 beta samples = [0.0,1.9]
dfreq = (f2-f1)/nfreq      # freq step
db = 0.05                  # beta step

# -- two approaches to peak scaling
fx = linspace(f1,f2, nfreq)
c1 = scalingTable()        # 2D lookup table for max values
c2 = scalingCoeffs()       # polynomial and power approximations

# -- 1: 2D lookup table, 2: polynomial/power approximations
setScalingMode(1)

y = fbam(f0,B)
y = scale(y,stab,f0,B)
spec = spectrum(y)


# -- plotting
figure(facecolor = '#e2e2e2')
fontcolor = '#222222'
font = FontProperties(family='sans-serif', size=9)

p1 = subplot(211)
ax1, = p1.plot(y, 'k')
grid(True)
xlim(0,4*int(fs/f0))
ylim(-1.1,1.1)

labels = p1.get_xticklabels() + p1.get_yticklabels()
setp(labels, color=fontcolor, fontproperties=font)
xlabel('Time (samples)', color=fontcolor, fontproperties=font)
ylabel('Level', color=fontcolor, fontproperties=font)

p2 = subplot(212)
ax2, = p2.plot(linspace(0,fs/2,len(spec)), spec, 'k')
grid(True)
xlim(0,fs/2)
ylim(-100,6)

labels = p2.get_xticklabels() + p2.get_yticklabels()
setp(labels, color=fontcolor, fontproperties=font)
xlabel('Freq (Hz)', color=fontcolor, fontproperties=font)
ylabel('Magnitude (dB)', color=fontcolor, fontproperties=font)

subplots_adjust(bottom=0.2)

# -- sliders
rect_B = axes([0.125, 0.10, 0.75, 0.03])
rect_f = axes([0.125, 0.05, 0.75, 0.03])
sb_B   = Slider(rect_B, 'B',  0.0,2.0, valinit=B)
sb_f   = Slider(rect_f, 'f0', f1, f2, valinit=f0)

def on_update(dummy):
   f0 = sb_f.val
   B = sb_B.val
   y = fbam(f0,B)
   y = scale(y,stab,f0,B)
   spec = spectrum(y)
   ax1.set_ydata(y)
   ax2.set_ydata(spec)
   draw()
sb_B.on_changed(on_update)
sb_f.on_changed(on_update)

suptitle('FBAM peak scaling', color=fontcolor, fontsize=12, family='serif')
show()