#
# fig 7 -- FBAM stability and aliasing
# (this takes about 17 minutes to compute on a 2 GHz laptop)
#
# Feedback Amplitude Modulation Synthesis
# J.Kleimola, V.Lazzarini, V.Vlimki, J.Timoney
# 2010
#
from numpy import *
from scipy.signal import *
from pylab import *
from time import time
from matplotlib.font_manager import FontProperties

fs = 44100.
t = linspace(0,44100,44100)

# -- cover 88-key piano keyboard
f1 = 27.5
f2 = 4186.01
P1 = int(ceil(fs/f1))
P2 = int(floor(fs/f2))


# -----------------------------------------------------------------------------
# straight FBAM theme
#
def theme(f0,B):
   w = 2*pi*f0/fs
   a = cos(w*t)      # coefficients
   x = a             # input
   #
   fbam = zeros(len(t))
   fbam[0] = 0
   for n in range(1,len(t)):
      fbam[n] = x[n] + B*a[n]*fbam[n-1]
   return fbam

# -----------------------------------------------------------------------------
# spectrum
#
NFFT = 16384
win = chebwin(NFFT, 120)
scal = NFFT*sqrt(mean(win**2))

def spectrum(sig):
   spec = fft(win*sig[0:NFFT])
   mags = sqrt(spec[0:NFFT/2].real**2 + spec[0:NFFT/2].imag**2)
   norm = 20*log10(mags/scal)
   norm = norm - max(norm)
   return norm


# -----------------------------------------------------------------------------
# stability
# returns beta that keeps the system stable
#
def beta_stability(N):
   gN = 1
   for n in range(1,N):
      gN *= cos(2*pi*n/N)
   # gN * B**(N-1) = 1
   B = pow(1./abs(gN), 1./(N-1))
   return B

# -----------------------------------------------------------------------------
# aliasing
# returns beta that keeps the aliasing below -80 dB
#
def beta_aliasing(f0):
   dB = 0.01
   # -- some defaults to boost computation time
   if   f0 < 600:    B = 1.5
   elif f0 < 1500:   B = 1.0
   elif f0 < 2300:   B = 0.7
   elif f0 < 3140:   B = 0.5
   else:             B = 0.2
   #
   mag = -1000
   while mag < -80:
      B += dB
      fbam = theme(f0,B)
      spec = spectrum(fbam)
      h = int((fs/2)/f0) + 1     # first aliasing harmonic
      fa = fs-h*f0               # highest folding frequency
      bin = int(fa*NFFT/fs)      # freq = bin*fs/NFFT
      mag = spec[bin]
      # numerical range may run out for very low fundamental frequencies
      # because they do not have aliasing problems. Use maximum beta
      if math.isinf(mag):
         B = 2+dB
         break
   return B-dB


start = time()

# -- stability
sp = arange(P1-2,0,-4)
sf = linspace(f1,f2,len(sp))
sb = zeros(len(sf))
sb.fill(float(inf))
si = 0
for N in sp:
   sb[si] = beta_stability(N)
   sf[si] = fs/N
   si += 1

# -- aliasing
NF = 100
bf = linspace(f1,f2,NF)
bb = zeros(NFFT/2)
bi = 0
for f in bf:
   bb[bi] = beta_aliasing(f)
   bi += 1

# -- aliasing (2x oversampling)
fs = 2 * fs
b2 = zeros(NFFT/2)
bi = 0
for f in bf:
   b2[bi] = beta_aliasing(f)
   bi += 1

print (time() - start) / 60.

# -- plotting
figure(figsize=(12,6), facecolor = '#e2e2e2')
font = FontProperties(family='Helvetica', size=9)
fontcolor = '#222222'

p1 = subplot(111)
hold(True)
p1.plot(bf,bb[0:len(bf)],'k')
p1.plot(bf,b2[0:len(bf)],'k-.')
p1.plot(sf,sb[0:len(sf)],'k--')
grid(True)
xlim(f1,f2)
ylim(0,2.05)
p1.yaxis.set_ticks(arange(0,2.01,0.2))

labels = p1.get_xticklabels() + p1.get_yticklabels()
setp(labels, color=fontcolor, fontproperties=font)
xlabel('Frequency (Hz, 88-key piano range)', color=fontcolor, fontproperties=font)
ylabel(r'$\beta$', color=fontcolor, fontproperties=font, size=14)

suptitle('FBAM stability (dashed) vs. aliasing (solid)', color=fontcolor, fontsize=10, family='Helvetica')
show()