#
# fig 9 -- FBAM scaling
# - 2D lookup table is created by fbam_scaling.py
# - power-law approximations are listed in fbam_scaling_gui.py
#
# Feedback Amplitude Modulation Synthesis
# J.Kleimola, V.Lazzarini, V.Vlimki, J.Timoney
# 2010
#
from scipy.signal import *
from pylab import *
from matplotlib.font_manager import FontProperties

# -- freq range is 88-key piano keyboard
f1 = 27.5
f2 = 4186.01
fs = 44100.


# -- 2D lookup table for scaling (generated by fbam_scaling.py)
def scalingTable():
   return genfromtxt("fbam_scale2D.txt")

# -- polynomial approximations
# -- these are used for betas 0..0.95
# -- for higher betas, see fbam_scaling_gui.py
def scalingCoeffs(fx, maxy, maxbeta=0.9, dbeta=0.1):
   coeffs = []
   ib = 0
   beta = 0.0
   eps = 0.0001   # python peculiarity
   while beta <= maxbeta+eps:
      if beta >= 0.9-eps:     k = 5
      elif beta >= 0.8-eps:   k = 3
      elif beta >= 0.7-eps:   k = 2
      else:                   k = 1
      func = polyfit(fx, maxy[:,ib], k)
      coeffs.append(func)
      beta += dbeta
      ib += 1
   return coeffs


# -- scaling
nfreq = 100                # 100 frequency samples (linear)
nbeta = 19                 # beta = [0.0,0.95]
db = 0.05                  # beta step

# -- two approaches to peak scaling
c1 = scalingTable()                          # 2D lookup table for max values

fx = linspace(f1, f2, nfreq)
coeffs = scalingCoeffs(fx,c1,0.95,0.05)      # polynomial approximations
c2 = zeros((nfreq,nbeta))
for i in range(0,nfreq):
   for b in range(0,nbeta):
      c2[i,b] = polyval(coeffs[b], fx[i])


# -- plotting
figure(figsize=(12,6), facecolor = '#e2e2e2')
fontcolor = '#222222'
font = FontProperties(family='sans-serif', size=9)

p2 = subplot(111)
for b in range(0,nbeta,2):
   p2.plot(fx,20*log10(c1[:,b]/10),color='0.5')
for b in range(0,nbeta,2):
   p2.plot(fx,20*log10(c2[:,b]/10),'k--')
grid(True)
xlim(f1,f2)
ylim(-21,1)

labels = p2.get_xticklabels() + p2.get_yticklabels()
setp(labels, color=fontcolor, fontproperties=font)
xlabel('Frequency (Hz, 88-key piano range)', color=fontcolor, fontproperties=font)
ylabel('Magnitude (dB)', color=fontcolor, fontproperties=font)

suptitle('FBAM scaling', color=fontcolor, fontsize=10, family='Helvetica')
show()