#
# fig 17 -- FBAM variation 5 : non-unitary feedback periods
#
# Feedback Amplitude Modulation Synthesis
# J.Kleimola, V.Lazzarini, V.Vlimki, J.Timoney
# 2010
#
from numpy import pi,cos
from scipy.signal import *
from pylab import *
from matplotlib.font_manager import FontProperties

f0 = 441.
fs = 44100.
w0 = 2*pi*f0/fs
N = int(fs/f0)
t = arange(0,44100*10)

a = cos(w0*t)     # coefficients
x = cos(w0*t)     # input
B = 0.85          # beta
D = 100           # feedback period (in samples)


# -- Delay line with linear interpolation
class Delay:
   def __init__(self, D, x0, maxlen=44100):
      self.D = int(D)
      self.frac = D - int(D)
      self.buf = zeros(maxlen)
      self.buf[0] = x0
      self.n = 0

   def getsample(self):
      s1 = self.buf[self.n]
      i2 = self.n + 1
      if (i2 > (self.D-1)): i2 = 0
      s2 = self.buf[i2]
      return s1 + self.frac*(s2 - s1)

   def putsample(self, sample):     
      self.buf[self.n] = sample
      self.n += 1
      if (self.n == self.D):
         self.n = 0


# -- variation 5
d = Delay(D,x[0])
y = zeros(len(t))
y[0] = 0
for n in range(1,len(t)):
   y[n] = x[n] + B*a[n] * d.getsample()
   d.putsample(y[n])
y = y / max(abs(y))

# -- and its spectrum
NFFT = 16384*2
NWIN = 16384
win = chebwin(NWIN, 120)
win = append(win, zeros(NFFT-NWIN))
scal = NFFT*sqrt(mean(win**2))
spec = fft(win*y[0:NFFT])
mags = sqrt(spec[0:NFFT/2].real**2 + spec[0:NFFT/2].imag**2)
norm = 20*log10(mags/scal)
spec = norm - max(norm)

# -- straight FBAM
fbam = zeros(len(t))
fbam[0] = 0
for n in range(1,len(t)):
   fbam[n] = x[n] + B*a[n]*fbam[n-1]
fbam = fbam / max(abs(fbam))

# -- and its spectrum
spec2 = fft(win*fbam[0:NFFT])
mags = sqrt(spec2[0:NFFT/2].real**2 + spec2[0:NFFT/2].imag**2)
norm = 20*log10(mags/scal)
spec2 = norm - max(norm)

# -- FBAM spectral envelope
sf = []
sm = []
f = 0
while f < fs/2:
   bin = round(f*NFFT/fs)
   sf.append(f)
   sm.append(spec2[bin])
   f += f0


# -- plotting
figure(facecolor='#e2e2e2')
fontcolor = '#222222'
font = FontProperties(family='sans-serif', size=9)

p1 = subplot(211)
p1.plot(t,y,'k',lw=1)
p1.plot(t,fbam,'k--',lw=1)
grid(True)
xlim(4410,4410+5*N)
ylim(-1.1,1.1)

labels = p1.get_xticklabels() + p1.get_yticklabels()
setp(labels, color=fontcolor, fontproperties=font)
xlabel('Time (samples)', color=fontcolor, fontproperties=font)
ylabel('Level', color=fontcolor, fontproperties=font)

p2 = subplot(212)
p2.plot(linspace(0,fs/2,len(spec)), spec, 'k', lw=1)
p2.plot(sf, sm, 'k--')
grid(True)
xlim(0,fs/2)
ylim(-100,6)

labels = p2.get_xticklabels() + p2.get_yticklabels()
setp(labels, color=fontcolor, fontproperties=font)
# p2.xaxis.set_ticklabels(["0","5","10","15","20"])
xlabel('Frequency (Hz)', color=fontcolor, fontproperties=font)
ylabel('Magnitude (dB)', color=fontcolor, fontproperties=font)

suptitle('FBAM variation 5: variable feedback periods', color=fontcolor, fontsize=10, family='Helvetica')
show()