#
# fig 21 -- FBAM AM/PM components
#
# Feedback Amplitude Modulation Synthesis
# J.Kleimola, V.Lazzarini, V.Vlimki, J.Timoney
# 2010
#
from pylab import *
from scipy import *
from scipy.signal import *
from matplotlib.font_manager import FontProperties

f0 = 441.
fs = 44100.
w0 = 2*pi*f0/fs
N = int(fs/f0)
L = 5*N
t = arange(L+N)

a = cos(w0*t)  # coefficients
x = cos(w0*t)  # input
B = 1          # beta


# -- straight FBAM
fbam = zeros(len(t))
fbam[0] = 0
for n in range(1,len(t)):
   fbam[n] = x[n] + B*a[n]*fbam[n-1]

# -- coefficients
def coeffs(a):
   # -- numerator
   h2D = [ones(L+N)]
   h2D.append(B*a[N:L+N])
   for i in range(2,N):
      h2D.append(B*a[N-i:L+N-i]*h2D[i-1])
   # -- denominator
   gN = prod(B*a[0:N])
   return h2D,gN

# -- frequency response of the filter
def freq_resp(n,w0,h2D,gN):
   num = complex(0.,0.)
   for i in range(0,len(h2D)):
      h = h2D[i]
      num += h[n]*exp(complex(0.,-w0*i))
   den = (1*exp(complex(0.,-w0*0)) - gN*exp(complex(0.,-w0*N)))
   return num/den


# -- get frequency response
h2D,gN = coeffs(a)
fr_mags = zeros(L)
fr_phas = zeros(L)
for n in t[0:L]:
   fres = freq_resp(n,w0,h2D,gN)
   fr_mags[n] = abs(fres)
   fr_phas[n] = arctan2(fres.imag, fres.real)

# -- generate signals using AM/PM
am = fr_mags*cos(w0*t[0:L])
pm = cos(w0*t[0:L]+fr_phas)

# -- plotting
fig = figure(figsize=(11,3), facecolor='#e2e2e2')
fontcolor = '#222222'
font = FontProperties(family='sans-serif', size=9)

p1 = subplot(121)
p1.plot(fbam/max(fbam),'k--')
p1.plot(am/max(am), 'k')
xlim(0,L)
ylim(-1.1,1.1)
grid(True)

labels = p1.get_xticklabels() + p1.get_yticklabels()
setp(labels, color=fontcolor, fontproperties=font)
xlabel('Time (samples)', color=fontcolor, fontproperties=font)
ylabel('Level', color=fontcolor, fontproperties=font)

p2 = subplot(122)
p2.plot(a, "k--")
p2.plot(pm, "k")
xlim(0,L)
ylim(-1.1,1.1)
grid(True)

labels = p2.get_xticklabels() + p2.get_yticklabels()
setp(labels, color=fontcolor, fontproperties=font)
xlabel('Time (samples)', color=fontcolor, fontproperties=font)
ylabel('Level', color=fontcolor, fontproperties=font)

suptitle('FBAM am/pm components', color=fontcolor, fontsize=10, family='Helvetica')
subplots_adjust(left=0.07, right=0.97, bottom=0.17)

show()