#
# fig 22 -- FBAM vs. PAF and ModFM
#
# Feedback Amplitude Modulation Synthesis
# J.Kleimola, V.Lazzarini, V.Vlimki, J.Timoney
# 2010
#
from pylab import *
from scipy import *
from scipy.signal import *
from matplotlib.font_manager import FontProperties

f0 = 500.
fs = 44100.
w0 = 2*pi*f0/fs
N = int(fs/f0)
L = 500*N
t = arange(L+N)


# -----------------------------------------------------------------------------
# spectrum
#
NFFT = 16384 / 2
win = chebwin(NFFT, 120)
scal = NFFT*sqrt(mean(win**2))

def spectrum(sig):
   spec = fft(win*sig[0:NFFT])
   mags = sqrt(spec[0:NFFT/2].real**2 + spec[0:NFFT/2].imag**2)
   norm = 20*log10(mags/scal)
   norm = norm - max(norm)
   return norm

# -----------------------------------------------------------------------------
# basic FBAM (difference equation)
#
def fbam(x,a,B):
   y = zeros(len(a))
   y[0] = 0
   for n in range(1,len(a)):
      y[n] = x[n] + B*a[n]*y[n-1]
   return y

# -----------------------------------------------------------------------------


# -- straight FBAM
w = 2*pi*f0/fs
a = cos(w*t)      # coefficients
x = a             # input
B = 1.62				# beta (max that keeps aliasing below -80 dB)
y = fbam(x,a,B)
y = y / max(y)
spec1 = spectrum(y)

# -- FBAM harmonic magnitudes (for contour in the plot)
sf1=[]
sm1=[]
f = 0
while f < fs/2:
   bin = round(f*NFFT/fs)
   sf1.append(f)
   sm1.append(spec1[bin])
   f += f0

# -- PAF (simple form: wm = n*wc, ws = 0)
def x(g):      return 2*sqrt(g)/(1-g)     # scaling (shaper input)
def p(g):      return (1+g)/(1-g)         # scaling (modulator)
def z(g,wm):   return x(g)*sin(wm*t/2.)   # scaled half sinusoid
def s(z):      return 1/(1+z**2)          # waveshaper
def M(g,z):    return p(g)*s(z)           # modulator
def C(wc):     return cos(wc*t)           # carrier

wc = w
wm = 1*wc
bw = 0.164			# corresponds to beta
g = exp(-wc/bw)

paf = M(g,z(g,wm)) * C(wc)
paf = paf / max(paf)
spec2 = spectrum(paf)

# -- PAF harmonic magnitudes (for contour in the plot)
sf2=[]
sm2=[]
f = 0
while f < fs/2:
   bin = round(f*NFFT/fs)
   sf2.append(f)
   sm2.append(spec2[bin])
   f += f0

# -- ModFM
wc = w
wm = w
k = 25
rm = exp(k*cos(wm*t)-k)
modfm = rm * cos(wc*t)
spec3 = spectrum(modfm)

# -- ModFM harmonic magnitudes (for contour in the plot)
sf3=[]
sm3=[]
f = 0
while f < fs/2:
   bin = round(f*NFFT/fs)
   sf3.append(f)
   sm3.append(spec3[bin])
   f += f0


# -----------------------------------------------------------------------------
# plotting
# -----------------------------------------------------------------------------

fig = figure(figsize=(11,3), facecolor = '#e2e2e2')
fontcolor = '#222222'
font = FontProperties(family='sans-serif', size=9)

p1 = subplot(121)
p1.plot(y[12:],'k--')    # shift left
p1.plot(modfm,color='0.55')
p1.plot(paf,'k')
xlim(0,3*N)
ylim(-0.2,1.1)
grid(True)

labels = p1.get_xticklabels() + p1.get_yticklabels()
setp(labels, color=fontcolor, fontproperties=font)
xlabel('Time (samples)', color=fontcolor, fontproperties=font)
ylabel('Level', color=fontcolor, fontproperties=font)

p2 = subplot(122)
p2.plot(sf3, sm3, '-', color='0.55')
p2.plot(sf1, sm1, '--', color='0.0')
p2.plot(linspace(0,fs/2,len(spec2)), spec2,'k')
p2.plot(sf2, sm2, 'k')
grid(True)
xlim(0,10000) #fs/2)
ylim(-80,6)

labels = p2.get_xticklabels() + p2.get_yticklabels()
setp(labels, color=fontcolor, fontproperties=font)
xlabel('Frequency (kHz)', color=fontcolor, fontproperties=font)
ylabel('Magnitude (dB)', color=fontcolor, fontproperties=font)
p2.xaxis.set_ticklabels(["0","2","4","6","8","10","12"])

suptitle('FBAM vs. PAF + ModFM', color=fontcolor, fontsize=10, family='Helvetica')
subplots_adjust(left=0.07, right=0.97, bottom=0.17)

show()