#
# fig 23 -- FBAM closed form approximation
#
# Feedback Amplitude Modulation Synthesis
# J.Kleimola, V.Lazzarini, V.Vlimki, J.Timoney
# 2010
#
from pylab import *
from scipy import *
from scipy.signal import *
from scipy.interpolate import splrep,splev
from matplotlib.font_manager import FontProperties

f0 = 441.
fs = 44100.
N = int(fs/f0)
L = 1000*N
t = arange(L+N)

# -----------------------------------------------------------------------------
# spectrum
#
NFFT = 16384
win = chebwin(NFFT, 120)
scal = NFFT*sqrt(mean(win**2))

def spectrum(sig):
   spec = fft(win*sig[0:NFFT])
   mags = sqrt(spec[0:NFFT/2].real**2 + spec[0:NFFT/2].imag**2)
   norm = 20*log10(mags/scal)
   norm = norm - max(norm)
   return norm

# -----------------------------------------------------------------------------
# basic FBAM (difference equation)
#
def fbam(x,a,B):
   y = zeros(len(a))
   y[0] = 0
   for n in range(1,len(a)):
      y[n] = x[n] + B*a[n]*y[n-1]
   return y

# -----------------------------------------------------------------------------


# -- straight FBAM
w = 2*pi*f0/fs
m = cos(w*t)      # coefficients
x = m             # input
B = 1             # beta
y = fbam(x,m,B)
y = y / max(y)
spec1 = spectrum(y)

# -- fbam harmonics (for contour in the plot)
sf1=[]
sm1=[]
f = 0
while f < fs/2:
   bin = round(f*NFFT/fs)
   sf1.append(f)
   sm1.append(spec1[bin])
   f += f0


# -- Jolley (593), sin of the numerator changed to cos
wc = 1*w    # theta
wm = 1*w    # beta
a = 2.5     # a**2 > 1 (or a=0.40 for a**2 < 1)
num = cos(wc*t)
den = (1 - 2*a*cos(wm*t) + a**2)**3
s4 = num / den
s = s4 / max(s4)
spec2 = spectrum(s)

# -- harmonics (for contour in the plot)
sf2=[]
sm2=[]
f = 0
while f < fs/2:
   bin = round(f*NFFT/fs)
   sf2.append(f)
   sm2.append(spec2[bin])
   f += f0

# -- waveshaper
phi = arange(0,1+f0/fs,f0/fs)
p = 2*phi - 1
num = p
den = (1 - 2*a*p + a**2)**3
g = num / den
g = g / max(g)
ws = splrep(p,g)
sws = splev(x, ws)
sws = sws / max(sws)


# -----------------------------------------------------------------------------
# plotting
# -----------------------------------------------------------------------------

fig = figure(figsize=(13,3),facecolor = '#e2e2e2')
fontcolor = '#222222'
font = FontProperties(family='sans-serif', size=9)

p1 = axes([0.05, 0.17, 0.18, 0.73])
p1.plot(y[5:], 'k--')
p1.plot(s, 'k')
xlim(0,2*N)
ylim(-0.2,1.1)
grid(True) 

labels = p1.get_xticklabels() + p1.get_yticklabels()
setp(labels, color=fontcolor, fontproperties=font)
xlabel('Time (samples)', color=fontcolor, fontproperties=font)
ylabel('Level', color=fontcolor, fontproperties=font)

p2 = axes([0.32, 0.17, 0.18, 0.73])
p2.plot(sf1, sm1, 'k--')
p2.plot(linspace(0,fs/2,len(spec2)), spec2, 'k')
grid(True)
xlim(0,10000)
ylim(-100,6)

labels = p2.get_xticklabels() + p2.get_yticklabels()
setp(labels, color=fontcolor, fontproperties=font)
xlabel('Frequency (kHz)', color=fontcolor, fontproperties=font)
ylabel('Magnitude (dB)', color=fontcolor, fontproperties=font)
p2.xaxis.set_ticklabels(["0","2","4","6","8","10"])

p3 = axes([0.58, 0.17, 0.18, 0.73])
p3.plot(p,g,'k')
p3.plot(cos(2*pi*phi),p,color='0.7')
p3.plot(p,sws[:len(p)],'k--')
grid(True)
xlim(-1.1,1.1)
ylim(-1.1,1.1)

labels = p3.get_xticklabels() + p3.get_yticklabels()
setp(labels, color=fontcolor, fontproperties=font)
xlabel('Input' , color=fontcolor, fontproperties=font)
ylabel('Output', color=fontcolor, fontproperties=font)


suptitle('FBAM closed form approximation', color=fontcolor, fontsize=10, family='Helvetica')
subplots_adjust(left=0.07, right=0.97, bottom=0.17)

show()