// ----------------------------------------------------------------------------
// dpw~ -- Differentiated Polynomial Waveforms
// 
// Implements the Differentiated Polynomial Waveforms sawtooth published in
// [1] Vlimki: IEEE SPL, vol.12, no.3 (March 2005)
// [2] Vlimki,Nam,Smith,Abel: IEEE Trans. ASL, vol.18, no.4, (May 2010)
//
// Jari Kleimola 2011
//
#define __attribute__(x)	;
#include "m_pd.h"
#include <string.h>
#include <math.h>
#include <stdlib.h>

#define MAXW 3
#define MAXVOICES 128
#define MAXSETS 20

typedef struct
{
	double m_c;				// scaling
	double m_z[MAXW];		// delays
	double m_f0;
	double m_T0;
	double m_phase;
	int i;
} Voice;

static t_class* dpw_class;
typedef struct _DPW
{
	t_object x_obj;
	t_float x_x;
	int m_W;
	void (*m_process)(Voice*, t_float*, int);		// dsp
	Voice* m_voices[MAXSETS][MAXVOICES];
	int m_noteStart;
	int m_noteEnd;
	double m_polyphonyScaler;
	t_float* m_audiobuf;
} DPW;

static void dpw_setW(DPW* dpw, t_floatarg W);
static void noteCluster(DPW* dpw, int m, int start, int numNotes);
static int clip(int i, int min, int max);
static float f0 = 0;


#pragma region ctor / dtor

// ----------------------------------------------------------------------------
// ctor
//
static void* dpw_new(t_floatarg W)
{
	int m,n;
	DPW* dpw = (DPW*)pd_new(dpw_class);

	// -- iolets
	inlet_new(&dpw->x_obj, &dpw->x_obj.ob_pd, &s_float, gensym("W"));
	outlet_new(&dpw->x_obj, &s_signal);

	// -- init
	for (m=0; m<MAXSETS; m++)
	{
		for (n=0; n<MAXVOICES; n++)
			dpw->m_voices[m][n] = 0;
		noteCluster(dpw, m, 21, 88);
	}
	// -- 88 key piano keyboard:
	// -- A0 = MIDI 21  = 27.5 Hz
	// -- C8 = MIDI 108 = 4186.01 Hz
	dpw->m_noteStart = 21; 
	dpw->m_noteEnd = 21 + 88;
	dpw->m_polyphonyScaler = 1./(20*88);
	dpw->m_audiobuf = (t_float*)malloc(sys_getblksize());
	dpw_setW(dpw, W);
	
	return dpw;
}

// ----------------------------------------------------------------------------
// dtor
//
static void dpw_dtor(DPW* dpw)
{
	int m,n;
	for (m=0; m<MAXSETS; m++)
	{
		for (n=0; n<MAXVOICES; n++)
			if (dpw->m_voices[m][n]) free(dpw->m_voices[m][n]);
	}
	free(dpw->m_audiobuf);
}

static void noteCluster(DPW* dpw, int set, int start, int numNotes)
{
	int n;

	start = clip(start, 0,127);
	numNotes = clip(numNotes, 0,128);
	if (start + numNotes > MAXVOICES)
		numNotes = MAXVOICES - start;

	for (n=start; n<start+numNotes; n++)
	{
		double f0 = 440 * pow(2.,(n-69)/12.);
		Voice* v = (Voice*)malloc(sizeof(Voice));
		v->m_f0 = f0;
		v->m_T0 = f0 / sys_getsr();
		v->m_phase = 0;
		dpw->m_voices[set][n] = v;
	}
}

#pragma endregion

#pragma region DSP

// ----------------------------------------------------------------------------
// main DSP routine
//
static t_int* dpw_process(t_int* w)
{
	DPW* dpw = (DPW*)w[1];
	t_float* out = (t_float*)w[2];
	int vecsize = (int)w[3];
	int m,n,v;
	double c;
	t_float* p,f;

	p = dpw->m_audiobuf;
	for (n=0; n<vecsize; n++)
		*p++ = 0.f;

	// -- dispatch
	for (m=0; m<MAXSETS; m++)
	{
		for (v = dpw->m_noteStart; v < dpw->m_noteEnd; v++)
			(*dpw->m_process)(dpw->m_voices[m][v], dpw->m_audiobuf, vecsize);
	}

	// -- mix to out
	c = dpw->m_polyphonyScaler;
	p = dpw->m_audiobuf;
	for (; vecsize>0; vecsize--)
		*out++ = (*p++) * c;

	return (w+4);
}

// ----------------------------------------------------------------------------
// trivial bipolar ramp (not using DPW, here just for reference)
//
static void dpw_trivial(Voice* voice, t_float* out, int vecsize)
{
	double x  = voice->m_phase;
	double T0 = voice->m_T0;
	int n;

	for (n=0; n<vecsize; n++)
	{
		*out++ += 2*x - 1;
		x += T0;
		if (x > 1) x -= 1.0;
	}

	voice->m_phase = x;
}

// ----------------------------------------------------------------------------
// W=1 (DPW order N=2)
// this is the 'original' DPW method published in [1]
//
static void dpw_1(Voice* voice, t_float* out, int vecsize)
{
	double x = voice->m_phase;
	double T0 = voice->m_T0;
	double z = voice->m_z[0];
	double c = voice->m_c;
	double y;
	int n;

	for (n = 0; n<vecsize; n++)
	{
		// -- trivial bipolar ramp
		double s = 2*x - 1;
		x += T0;
		if (x > 1) x -= 1;

		// -- parabolic DPW (s^2)
		s = s*s;
		y = s - z;
		z = s;
		*out++ += y * c;
	}

	voice->m_z[0] = z;
	voice->m_phase = x;
}

// ----------------------------------------------------------------------------
//  W=2 (DPW order N=3)
//
static void dpw_2(Voice* voice, t_float* out, int vecsize)
{
	double x = voice->m_phase;
	double z0 = voice->m_z[0];
	double z1 = voice->m_z[1];
	double T0 = voice->m_T0;
	double c = voice->m_c;
	double y0,y1;
	int n;

	for (n=0; n<vecsize; n++)
	{
		// -- trivial bipolar ramp
		double s = 2*x - 1;
		x += T0;
		if (x > 1) x -= 1;

		// -- DPW (s^3 - s)
		s = s*s*s - s;
		y0 =  s - z0;
		y1 = y0 - z1;
		z0 = s;
		z1 = y0;
		*out++ += y1 * c;
	}

	voice->m_z[0] = z0;
	voice->m_z[1] = z1;
	voice->m_phase = x;
}

// ----------------------------------------------------------------------------
// W=3 (DPW order N=4)
//
static void dpw_3(Voice* voice, t_float* out, int vecsize)
{
	double x = voice->m_phase;
	double z0 = voice->m_z[0];
	double z1 = voice->m_z[1];
	double z2 = voice->m_z[2];
	double T0 = voice->m_T0;
	double c = voice->m_c;
	double y0,y1,y2,s2;
	int n;

	for (n=0; n<vecsize; n++)
	{
		// -- trivial bipolar ramp
		double s = 2*x - 1;
		x += T0;
		if (x > 1) x -= 1;

		// -- DPW (s^4 - 2s^2)
		// -- ref [2] counts 3 multiplications, but it's possible to manage with 2
		s2 = s*s;
		s  = s2*(s2 - 2);
		y0 =  s - z0;
		y1 = y0 - z1;
		y2 = y1 - z2;
		z0 = s;
		z1 = y0;
		z2 = y1;
		*out++ += y2 * c;
	}

	voice->m_z[0] = z0;
	voice->m_z[1] = z1;
	voice->m_z[2] = z2;
	voice->m_phase = x;
}

// ----------------------------------------------------------------------------
// called when DSP is turned on. Registers the actual processing method.
//
static void dpw_dsp(DPW* dpw, t_signal** sp)
{
	dsp_add(dpw_process, 3, dpw, sp[0]->s_vec, sp[0]->s_n);
}

#pragma endregion

#pragma region utils

static int clip(int i, int min, int max)
{
	if (i < min) i = min;
	else if (i > max)	i = max;
	return i;
}

// ----------------------------------------------------------------------------
// scaling factor depends on f0 and N (helper function)
//
static void dpw_updateScaling(DPW* dpw, Voice* voice)
{	
	// -- scaling factors (nominator)
	static double s_cn[] = {1,4,24,192,1920,23040};

	if (dpw->m_W == 0) voice->m_c = 1;
	else
	{
		int w;
		double P = 1/voice->m_T0;		// period
		double S = P;
		for (w=2; w<=dpw->m_W; w++)	// P ^ (N-1)
			S *= P;
		voice->m_c = S / s_cn[dpw->m_W];
	}
}

#pragma endregion

#pragma region input parameters

// ----------------------------------------------------------------------------
// DPW polynomial order N = W + 1
//
static void dpw_setW(DPW* dpw, t_floatarg W)
{
	int m,n,i;

	dpw->m_W = clip((int)W, 0,3);
	switch (dpw->m_W)
	{
		case 0:	dpw->m_process = dpw_trivial;	break;
		case 1:	dpw->m_process = dpw_1;			break;
		case 2:	dpw->m_process = dpw_2;			break;
		case 3:	dpw->m_process = dpw_3;			break;
	}

	// -- update state
	for (m=0; m<MAXSETS; m++)
	{
		for (n = dpw->m_noteStart; n < dpw->m_noteEnd; n++)
		{
			dpw_updateScaling(dpw, dpw->m_voices[m][n]);
			for (i=0; i<MAXW; i++)
				dpw->m_voices[m][n]->m_z[i] = 0;
		}
	}
}


#pragma endregion

// ----------------------------------------------------------------------------
// entry point 
//
void dpw_tilde_setup(void)
{
	dpw_class = class_new(gensym("dpw~"), (t_newmethod)dpw_new, (t_method)dpw_dtor, sizeof(DPW), CLASS_NOINLET, A_DEFFLOAT, 0);
	class_addmethod(dpw_class, (t_method)dpw_dsp, gensym("dsp"), (t_atomtype)0);
	class_addmethod(dpw_class, (t_method)dpw_setW, gensym("W"), A_FLOAT, 0);
}
