// ----------------------------------------------------------------------------
// ptr~ -- Polynomial Transition Regions
// 
// Implements the Polynomial Transition Regions sawtooth
//
// Jari Kleimola 2011
//
#define __attribute__(x)	;
#include "m_pd.h"
#include <string.h>
#include <math.h>
#include <stdlib.h>

#define MAXW 3
#define MAXVOICES 128
#define MAXSETS 20

typedef struct
{
	double m_a31;
	double m_a21;
	double m_a11;
	double m_a01;
	double m_a32;
	double m_a22;
	double m_a12;
	double m_a02;
	double m_a33;
	double m_a23;
	double m_a13;
	double m_a03;

	double m_f0;
	double m_P0;
	double m_T0;
	double m_T2;
	double m_T3;
	double m_phase;
	int i;
} Voice;

static t_class* ptr_class;
typedef struct _PTR
{
	t_object x_obj;
	t_float x_x;
	double m_h;
	int m_W;
	void (*m_process)(Voice*, t_float*, int);		// dsp
	Voice* m_voices[MAXSETS][MAXVOICES];
	int m_noteStart;
	int m_noteEnd;
	double m_polyphonyScaler;
	t_float* m_audiobuf;
} PTR;

static void ptr_setW(PTR* ptr, t_floatarg W);
static void ptr_updateCoeffs(PTR* ptr, Voice* voice);
static void noteCluster(PTR* ptr, int m, int start, int numNotes);
static int clip(int i, int min, int max);
static float f0 = 0;


#pragma region ctor / dtor

// ----------------------------------------------------------------------------
// ctor
//
static void* ptr_new(t_floatarg W)
{
	int m,n;
	int start = 21;
	int numNotes = 88;
	PTR* ptr = (PTR*)pd_new(ptr_class);

	// -- iolets
	inlet_new(&ptr->x_obj, &ptr->x_obj.ob_pd, &s_float, gensym("W"));
	outlet_new(&ptr->x_obj, &s_signal);

	// -- init
	ptr->m_h = 1;
	for (m=0; m<MAXSETS; m++)
	{
		for (n=0; n<MAXVOICES; n++)
			ptr->m_voices[m][n] = 0;
		noteCluster(ptr, m, start, numNotes);
	}
	// -- 88 key piano keyboard:
	// -- A0 = MIDI 21  = 27.5 Hz
	// -- C8 = MIDI 108 = 4186.01 Hz
	ptr->m_noteStart = start;
	ptr->m_noteEnd = start + numNotes;

	ptr->m_polyphonyScaler = 1./(MAXSETS*numNotes);
	ptr->m_audiobuf = (t_float*)malloc(sys_getblksize());
	ptr_setW(ptr, W);

	return ptr;
}

// ----------------------------------------------------------------------------
// dtor
//
static void ptr_dtor(PTR* ptr)
{
	int m,n;
	for (m=0; m<MAXSETS; m++)
	{
		for (n=0; n<MAXVOICES; n++)
			if (ptr->m_voices[m][n]) free(ptr->m_voices[m][n]);
	}
	free(ptr->m_audiobuf);
}

static void noteCluster(PTR* ptr, int set, int start, int numNotes)
{
	int n;

	start = clip(start, 0,127);
	numNotes = clip(numNotes, 0,128);
	if (start + numNotes > MAXVOICES)
		numNotes = MAXVOICES - start;

	for (n=start; n<start+numNotes; n++)
	{
		double f0 = 440 * pow(2.,(n-69)/12.);
		Voice* v = (Voice*)malloc(sizeof(Voice));
		v->m_f0 = f0;
		v->m_T0 = f0 / sys_getsr();
		v->m_T2 = 2 * v->m_T0;
		v->m_T3 = 3 * v->m_T0;
		v->m_P0 = 1 / v->m_T0;
		v->m_phase = 0;
		// ptr_updateCoeffs(ptr, v);
		ptr->m_voices[set][n] = v;
		v->i = n;
	}
}

#pragma endregion

#pragma region DSP

// ----------------------------------------------------------------------------
// main DSP routine
//
static t_int* ptr_process(t_int* w)
{
	PTR* ptr = (PTR*)w[1];
	t_float* out = (t_float*)w[2];
	int vecsize = (int)w[3];
	int m,n,v;
	double c;
	float f,y;
	t_float* p;

	p = ptr->m_audiobuf;
	for (n=0; n<vecsize; n++)
		*p++ = 0.f;

	// -- dispatch
	for (m=0; m<MAXSETS; m++)
	{
		for (v = ptr->m_noteStart; v < ptr->m_noteEnd; v++)
			(*ptr->m_process)(ptr->m_voices[m][v], ptr->m_audiobuf, vecsize);
	}

	// -- mix to out
	c = ptr->m_polyphonyScaler;
	p = ptr->m_audiobuf;
	for (; vecsize>0; vecsize--)
	{
		*out++ = (*p++) * c;
	}

	return (w+4);
}

// ----------------------------------------------------------------------------
// trivial bipolar ramp (not using PTR, here just for reference)
//
static void ptr_trivial(Voice* voice, t_float* out, int vecsize)
{
	double x  = voice->m_phase;
	double T0 = voice->m_T0;
	int n;

	for (n=0; n<vecsize; n++)
	{
		*out++ += (t_float)(2*x - 1);
		x += T0;
		if (x > 1) x -= 1.0;
	}

	voice->m_phase = x;
}

// ----------------------------------------------------------------------------
// W=1 (PTR order N=2)
// this is the 'original' PTR method published in [1]
//
static void ptr_1(Voice* voice, t_float* out, int vecsize)
{
	double x  = voice->m_phase;
	double T0 = voice->m_T0;
	double a11 = voice->m_a11;
	double a01 = voice->m_a01;
	int n;

	for (n = 0; n<vecsize; n++)
	{
		// -- ptr
		if (x >= T0)	*out++ +=  2*x - 1;
		else				*out++ += a11*x - a01;

		// -- unipolar modulo counter
		x += T0;
		if (x > 1) x -= 1;
	}

	voice->m_phase = x;
}

// ----------------------------------------------------------------------------
//  W=2 (PTR order N=3)
//
static void ptr_2(Voice* voice, t_float* out, int vecsize)
{
	double x  = voice->m_phase;
	double P0 = voice->m_P0;
	double T0 = voice->m_T0;
	double T2 = voice->m_T2;
	double a21 = voice->m_a21;
	double a11 = voice->m_a11;
	double a01 = voice->m_a01;
	double a22 = voice->m_a22;
	double a12 = voice->m_a12;
	double a02 = voice->m_a02;
	double D;
	int n;

	for (n=0; n<vecsize; n++)
	{
		// -- ptr
		if (x >= T2)
			*out++ +=  2*x - 1;
		else if (x >= T0)
		{
			D = x * P0;
			*out++ +=  (a22*D + a12)*D + a02;
		}
		else
		{
			D = x * P0;
			*out++ +=  (a21*D + a11)*D + a01;
		}

		// -- unipolar modulo counter
		x += T0;
		if (x > 1) x -= 1;
	}

	voice->m_phase = x;
}

// ----------------------------------------------------------------------------
// W=3 (PTR order N=4)
//
static void ptr_3(Voice* voice, t_float* out, int vecsize)
{
	double x = voice->m_phase;
	double P0 = voice->m_P0;
	double T0 = voice->m_T0;
	double T2 = voice->m_T2;
	double T3 = voice->m_T3;
	double a31 = voice->m_a31;
	double a11 = voice->m_a11;
	double a01 = voice->m_a01;
	double a32 = voice->m_a32;
	double a22 = voice->m_a22;
	double a12 = voice->m_a12;
	double a02 = voice->m_a02;
	double a33 = voice->m_a33;
	double a23 = voice->m_a23;
	double a13 = voice->m_a13;
	double a03 = voice->m_a03;
	double D;
	int n;

	for (n=0; n<vecsize; n++)
	{
		// -- ptr
		if (x >= T3)
			*out++ +=  2*x - 1;
		else if (x >= T2)
		{
			D = x * P0;
			*out++ += ((a33*D + a23)*D + a13)*D + a03;
		}
		else if (x >= T0)
		{
			D = x * P0;
			*out++ += ((a32*D + a22)*D + a12)*D + a02;
		}
		else
		{
			D = x * P0;
			*out++ +=  (a31*D*D + a11)*D + a01;
		}

		// -- unipolar modulo counter
		x += T0;
		if (x > 1) x -= 1;
	}

	voice->m_phase = x;
}

// ----------------------------------------------------------------------------
// called when DSP is turned on. Registers the actual processing method.
//
static void ptr_dsp(PTR* ptr, t_signal** sp)
{
	dsp_add(ptr_process, 3, ptr, sp[0]->s_vec, sp[0]->s_n);
}

#pragma endregion

#pragma region utils

static int clip(int i, int min, int max)
{
	if (i < min) i = min;
	else if (i > max)	i = max;
	return i;
}

// ----------------------------------------------------------------------------
// (helper function)
//
static void ptr_updateCoeffs(PTR* ptr, Voice* voice)
{
	double h = ptr->m_h;
	double T0 = voice->m_T0;
	double T2 = voice->m_T2;
	double T3 = voice->m_T3;

	switch (ptr->m_W)
	{
		case 1:
		{
			double P0 = voice->m_P0;
			voice->m_a11 = 2 - 2*h*P0;
			voice->m_a01 = 2*h - 1 - T0;
			break;
		}
		case 2:
			voice->m_a21 = -h;
			voice->m_a11 = T2;
			voice->m_a01 = 2*h - 1 - T2;
			voice->m_a22 = h;
			voice->m_a12 = T2 - 4*h;
			voice->m_a02 = 4*h - 1 - T2;
			break;
		case 3:
			voice->m_a31 = -h/3.;
			voice->m_a11 = T2;
			voice->m_a01 = 2*h - 1 - T3;
			voice->m_a32 = -2*voice->m_a31;
			voice->m_a22 = -3*h;
			voice->m_a12 = T2 - voice->m_a22;
			voice->m_a02 = h - 1 - T3;
			voice->m_a33 = voice->m_a31;
			voice->m_a23 = -voice->m_a22;
			voice->m_a13 = T2 - 9*h;
			voice->m_a03 = 9*h - 1 - T3;
			break;
	}
}

#pragma endregion

#pragma region input parameters

// ----------------------------------------------------------------------------
// PTR transition region width
//
static void ptr_setW(PTR* ptr, t_floatarg W)
{
	int m,n,i;

	ptr->m_W = clip((int)W, 0,3);
	switch (ptr->m_W)
	{
		case 0:	ptr->m_process = ptr_trivial;	break;
		case 1:	ptr->m_process = ptr_1;			break;
		case 2:	ptr->m_process = ptr_2;			break;
		case 3:	ptr->m_process = ptr_3;			break;
	}

	// -- update state
	for (m=0; m<MAXSETS; m++)
	{
		for (n = ptr->m_noteStart; n < ptr->m_noteEnd; n++)
			ptr_updateCoeffs(ptr, ptr->m_voices[m][n]);
	}
}

#pragma endregion

// ----------------------------------------------------------------------------
// entry point 
//
void ptr_tilde_setup(void)
{
	ptr_class = class_new(gensym("ptr~"), (t_newmethod)ptr_new, (t_method)ptr_dtor, sizeof(PTR), CLASS_NOINLET, A_DEFFLOAT, 0);
	class_addmethod(ptr_class, (t_method)ptr_dsp, gensym("dsp"), (t_atomtype)0);
	class_addmethod(ptr_class, (t_method)ptr_setW, gensym("W"), A_FLOAT, 0);
}
