# =============================================================================
# PTR sawtooth signals
# =============================================================================
from scipy.signal import *
from scipy import *
from pylab import *
from matplotlib.font_manager import FontProperties


f0 = 2489.		# midi note 99 (D#7)
fs = 44100.
T0 = f0/fs
P0 = fs/f0

t = arange(0,1*fs)
L = len(t)

# -----------------------------------------------------------------------------
# (1) bipolar modulo counter
#
def phi(T0,p0=0):
	s = zeros(L)
	p = p0
	for n in range(0,L):
		s[n] = p
		p += T0
		if p > 1: p -= 1
	return s

# -----------------------------------------------------------------------------
# (2) trivial saw
#
def saw(T0):
	s = 2*phi(T0,0.5) - 1
	return s

# -----------------------------------------------------------------------------
# (Table 2) PTR W=1
#
# operations per	period	sample
# - multiply		P			1
# - add				P			1
# - branch			P			1
#
def PTR1(T0):
	h = 1										# f0 update for h=1:
	cdc = T0									# m
	DC = 1 + cdc
	# -- coefficients (5 operations)
	a1 = 2 - 2*h*P0		# am			# am
	a0 = 2*h - DC			# maa			# a
	#
	p = phi(T0,0.5)
	y = zeros(L)
	for n in range(0,L):
		if p[n] >= T0:	y[n] =  2*p[n] - DC	# outside	MA
		else:				y[n] = a1*p[n] + a0	# inside		MA
	return y

# -----------------------------------------------------------------------------
# (Table 2) PTR W=2
#
# operations per	period	sample
# - multiply		P+4		1+4T
# - add				P+2		1+2T
# - branch			P+1		1+T
#
def PTR2(T0):
	h = 1										# f0 update for h=1:
	T2 = T0+T0								# ma
	cdc = T2
	DC = 1 + cdc
	# -- coefficients (7 operations)
	a21 = -h
	a11 = T2					# a
	a01 = 2*h - DC			# maa			# a
	a22 = h
	a12 = T2 - 4*h			# am			# a
	a02 = 4*h - DC			# a			# a
	#
	p = phi(T0,0.5)
	y = zeros(L)
	for n in range(0,L):
		if p[n] >= T2:								# outside
			y[n] = 2*p[n] - DC					# MA
		elif p[n] >= T0:							# inside
			D = p[n] * P0							# M
			y[n] = (a22*D + a12)*D + a02		# MAMA
		else:											# inside
			D = p[n] * P0							# M
			y[n] = (a21*D + a11)*D + a01		# MAMA
	return y

# -----------------------------------------------------------------------------
# (Table 2) PTR W=3
#
# operations per	period	sample
# - multiply		P+9		1+9T
# - add				P+5		1+5T
# - branch			P+2		1+2T
#
def PTR3(T0):
	h = 1										# f0 update for h=1:
	T2 = T0+T0								# ma
	T3 = T2+T0								# a
	cdc = T3
	DC = 1 + cdc
	# -- coefficients (13 operations)
	a31 = -h/3.
	a11 = T2					# a
	a01 = 2*h - DC			# maaa		# a
	a32 = 2*h/3.			# m
	a22 = -3*h				# m
	a12 = T2 - a22			# a			# a
	a02 = h - DC			# a			# a
	a33 = a31
	a23 = -a22				# a
	a13 = T2 - 9*h			# am			# a
	a03 = 9*h - DC			# a			# a
	#
	p = phi(T0,0.5)
	y = zeros(L)
	for n in range(0,L):
		if p[n] >= T3:											# outside
			y[n] = 2*p[n] - DC								# MA
		elif p[n] >= T2:										# inside
			D = p[n] * P0										# M
			y[n] = ((a33*D + a23)*D + a13)*D + a03		# MAMAMA
		elif p[n] >= T0:										# inside
			D = p[n] * P0										# M
			y[n] = ((a32*D + a22)*D + a12)*D + a02		# MAMAMA
		else:														# inside
			D = p[n] * P0										# M
			y[n] = (a31*D*D + a11)*D + a01				# MMAMA
	return y


# =============================================================================
# main
# =============================================================================

# -- PTR sawtooths
s0 = saw(T0)
s1 = PTR1(T0)
s2 = PTR2(T0)
s3 = PTR3(T0)


# =============================================================================
# plotting
# =============================================================================

fig = figure(figsize=(10,8), facecolor='#e2e2e2')
font = FontProperties(family='serif', size=12)
PL = 3*P0
fig.canvas.set_window_title('PTR')
suptitle('PTR sawtooth signals, $f_0$= 2489 Hz, $f_s$= 44100 Hz', fontproperties=font)

# -- helper
def set_waveplot(p,label, marker=None,stem=None,base=None):
	grid(True)
	xlim(0,PL)
	ylim(-1.1,1.1)
	xlabel('Time (samples)', fontproperties=font)
	ylabel(label, fontproperties=font, rotation=90)
	labels = p.get_xticklabels() + p.get_yticklabels()
	setp(labels, fontproperties=font)
	if marker:	setp(marker, 'markersize', 6)
	if stem:		setp(stem, 'alpha','0')
	if base:		setp(base, 'color', 'None')

# -- helper
def set_specplot(p):
	grid(True)
	xlim(0,fs/2)
	ylim(-80,6)
	xlabel('Frequency (kHz)', fontproperties=font)
	ylabel('Magnitude (dB)', fontproperties=font)
	labels = p.get_xticklabels() + p.get_yticklabels()
	setp(labels, fontproperties=font)
	p.xaxis.set_ticklabels(["0","5","10","15","20"])
	p.yaxis.set_ticks([0,-20,-40,-60,-80])

# -- magnitude spectrum
NFFT = 16384*2
NWIN = 16384
win = chebwin(NWIN, 120)
win = append(win, zeros(NFFT-NWIN))
scal = NFFT*sqrt(mean(win**2))
def spectrum(s):
	spec = fft(win*s[0:NFFT])
	mags = sqrt(spec[0:NFFT/2].real**2 + spec[0:NFFT/2].imag**2)
	norm = 20*log10(mags/scal)
	spec = norm - max(norm)
	return spec


p11 = subplot(421)
p11.plot(s0, 'k', lw=1)
marker,stem,base = p11.stem(t[0:PL], s0[0:PL], linefmt='k-', markerfmt='ko')
set_waveplot(p11,'trivial', marker,stem,base)

m0  = spectrum(s0)
p12 = subplot(422)
p12.plot(linspace(0,fs/2,len(m0)), m0, 'k', lw=1)
set_specplot(p12)

p21 = subplot(423)
p21.plot(s1, 'k', lw=1)
marker,stem,base = p21.stem(t[0:PL], s1[0:PL], linefmt='k-', markerfmt='ko')
set_waveplot(p21,'N=2', marker,stem,base)

m1  = spectrum(s1)
p22 = subplot(424)
p22.plot(linspace(0,fs/2,len(m1)), m1, 'k', lw=1)
set_specplot(p22)

p31 = subplot(425)
p31.plot(s2, 'k', lw=1)
marker,stem,base = p31.stem(t[0:PL], s2[0:PL], linefmt='k-', markerfmt='ko')
set_waveplot(p31,'N=3', marker,stem,base)

m2  = spectrum(s2)
p32 = subplot(426)
p32.plot(linspace(0,fs/2,len(m2)), m2, 'k', lw=1)
set_specplot(p32)

p41 = subplot(427)
p41.plot(s3, 'k', lw=1)
marker,stem,base = p41.stem(t[0:PL], s3[0:PL], linefmt='k-', markerfmt='ko')
set_waveplot(p41,'N=4', marker,stem,base)

m3  = spectrum(s3)
p42 = subplot(428)
p42.plot(linspace(0,fs/2,len(m3)), m3, 'k', lw=1)
set_specplot(p42)


fig.subplots_adjust(left=0.1, right=0.92, top=0.94, bottom=0.07, wspace=0.27, hspace=0.33)
show()
