# =============================================================================
# Fig. 2 -- Computational Load
# =============================================================================

from scipy.signal import *
from scipy import *
from pylab import *
from matplotlib.font_manager import FontProperties

fs = 44100.
freqs = arange(0,fs/2,100)
L = len(freqs)


# percentual savings
T0 = 4186/fs
PTR1 = 1		  + 1		  + 1;
PTR2 = 1+4*T0 + 1+2*T0 + 1+T0;
PTR3 = 1+9*T0 + 1+5*T0 + 1+2*T0;
DPW2 = 5.
DPW3 = 8.
DPW4 = 9.
print((DPW2-PTR1)/DPW2)
print((DPW3-PTR2)/DPW3)
print((DPW4-PTR3)/DPW4)


# -----------------------------------------------------------------------------
# DPW and trivial
# -----------------------------------------------------------------------------

TRIV = zeros(L)
DPW_N2 = zeros(L)
DPW_N3 = zeros(L)
DPW_N4 = zeros(L)

i = 0
for f in freqs:
	TRIV[i] = 2				# s = 2x-1
	DPW_N2[i] = 3 + 2		# s = 2x-1, s = s2, y = s/c, 1 diff
	DPW_N3[i] = 4 + 4		# s = 2x-1, s = s3-s, y = s/c, 2 diffs
	DPW_N4[i] =	4 + 5		# s = 2x-1, s = s4-2s2 = s2(s2-2), y = s/c, 3 diffs
	i += 1

# -----------------------------------------------------------------------------
# PTR
# -----------------------------------------------------------------------------

PTR_W1 = zeros(L)
PTR_W2 = zeros(L)
PTR_W3 = zeros(L)

i = 0
for f in freqs:
	T0 = f/fs
	PTR_W1[i] = 1		 + 1		 + 1;
	PTR_W2[i] = 1+4*T0 + 1+2*T0 + 1+T0;
	PTR_W3[i] = 1+9*T0 + 1+5*T0 + 1+2*T0;
	i += 1

# -----------------------------------------------------------------------------
# BLIT-FDF
# -----------------------------------------------------------------------------

FDF_W1 = zeros(L)
FDF_W2 = zeros(L)
FDF_W3 = zeros(L)

i = 0
for f in freqs:
	T0 = f/fs
	FDF_W1[i] = 1+2*T0  + 1+2*T0 + 1+T0;
	FDF_W2[i] = 1+4*T0  + 1+5*T0 + 1+2*T0;
	FDF_W3[i] = 1+10*T0 + 1+8*T0 + 1+3*T0;
	i += 1

# -----------------------------------------------------------------------------
# PolyBLEP
# -----------------------------------------------------------------------------

blepfreqs = arange(0,fs/2,750)
BLEP_W1 = zeros(L) - 1
BLEP_W2 = zeros(len(blepfreqs))
BLEP_W3 = zeros(L) - 1

i = 0
for f in blepfreqs:
	T0 = f/fs
	BLEP_W2[i] = 1+6*T0  + 1+7*T0 + 1+T0;
	i += 1

# -- plotting -----------------------------------------------------------------

def setPlotProps(p,s, yticks=True):
	xlim(0, 4186)
	ylim(0, 10)
	p.xaxis.set_ticklabels(["0","","1","","2","","3","","4"])
	if not yticks:
		p.yaxis.set_ticklabels([])
	labels = p.get_xticklabels() + p.get_yticklabels()
	setp(labels, fontproperties=font)
	xlabel(s, fontproperties=font)

fh = 4.5
fig = figure(figsize=(1.78*fh,fh), facecolor='#e2e2e2')
font = FontProperties(family='Times New Roman', size=21)
legendfont = FontProperties(family='Times New Roman', size=18)

p1 = subplot(131)
line1, = plot(freqs, DPW_N2, 'k--', lw=2)
line2, = plot(freqs, PTR_W1, 'k', lw=2)
line3, = plot(freqs, FDF_W1, 'k-.', lw=2)
line4, = plot(freqs, BLEP_W1, 'k-', lw=1)
setPlotProps(p1,'(a)', True)
ylabel('Operations per sample', fontproperties=font)
text(3700,-4, 'Fundamental frequency (kHz)', fontproperties=font)

p2 = subplot(132)
plot(freqs, DPW_N3, 'k--', lw=2)
plot(freqs, PTR_W2, 'k', lw=2)
plot(freqs, FDF_W2, 'k-.', lw=2)
plot(blepfreqs, BLEP_W2, 'kx', ms=8, mew=1)
plot(blepfreqs, BLEP_W2, 'k-', lw=1)
setPlotProps(p2,'(b)')

line4.set_marker('x')
line4.set_markersize(8)
line4.set_markeredgewidth(1)
leg = figlegend( (line1,line4,line3,line2), ('DPW','PolyBLEP ($W=2$)','BLIT-FDF','PTR'), loc=9, ncol=4,
	handlelength=2.3, handletextpad=0.1, prop=legendfont, columnspacing=1)
leg.draw_frame(False)

p3 = subplot(133)
plot(freqs, DPW_N4, 'k--', lw=2)
plot(freqs, PTR_W3, 'k', lw=2)
plot(freqs, FDF_W3, 'k-.', lw=2)
plot(freqs, BLEP_W3, ':', lw=0)
setPlotProps(p3,'(c)')

fig.subplots_adjust(top=0.86, bottom=0.31, left=0.12, right=0.95, wspace=0.32)
show()
