# =============================================================================
# Fig. 3 -- Transients
# =============================================================================

from scipy.signal import *
from scipy import *
from pylab import *
from matplotlib.font_manager import FontProperties
from matplotlib.patches import Ellipse
from scipy.io import wavfile

f0 = 500.
fm = 10.
fs = 44100
L = 2*fs


# -----------------------------------------------------------------------------
# DPW (W=2)
# -----------------------------------------------------------------------------
def dpw_sweep(f0,fm,fs):
	T0 = f0/fs
	z0 = z1 = 0
	TLFO = fm/fs
	y = zeros(L)
	x = 0.5
	a = 0
	#
	for n in range(0,L):
		# -- sweep
		T0 = (1+a)*f0/fs
		c = 1./(24*T0*T0)
		a += TLFO
		if a > 0.5:
			a = 0

		# -- DPW
		s  = 2*x - 1		# trivial saw
		s3 = s*s*s - s		# waveshaper
		y0 = s3 - z0		# diff
		y1 = y0 - z1		# diff
		y[n] = y1 * c		# scaling
		z0 = s3
		z1 = y0

		# -- phase
		x += T0
		if x > 1: x -= 1

	y[0] = y[1] = 0
	# y /= max(abs(y))		# scaling because of transients
	return y

# -----------------------------------------------------------------------------
# PTR (W=2)
# -----------------------------------------------------------------------------
def ptr_sweep(f0,fm,fs):
	y = zeros(L)
	x = 0.5 - f0/fs
	a = 0
	TLFO = fm/fs
	#
	for n in range(0,L):
		# -- sweep
		T0 = (1+a)*f0/fs
		T2 = 2*T0
		a += TLFO
		if a > 0.5:
			a = 0

		# -- PTR
		x += T0
		if x > 1:
			x -= 1
			P0 = 1/T0
			a21 = -P0*P0
			a01 = 2 - (1 + 2*T0)
			y[n] = (a21*x + 2)*x + a01
		elif x < T2:
			P0 = 1/T0
			a22 = P0*P0
			a12 = 2 - 4*P0
			a02 = 4 - (1 + T2)
			y[n] = (a22*x + a12)*x + a02
		else:
			y[n] = 2*x - (1 + T2)

	return y

# -----------------------------------------------------------------------------
# BLIT-FDF (W=2)
# -----------------------------------------------------------------------------
def fdf_sweep(f0,fm,fs):
	y = zeros(L)
	p = 0.5 * fs/f0
	a = 0
	TLFO = fm/fs
	#
	for n in range(0,L):
		# -- sweep
		T0 = (1+a)*f0/fs
		T2 = 2*T0
		a += TLFO
		if a > 0.5:
			a = 0

		# -- FDF
		if p >= 1.5:
			y[n] = -T2
		elif p >= 0.5:
			x = p - 1.5
			y[n] = x*x - T2
		elif p >= -0.5:
			y[n] = 1.5 - 2*p*p - T2
		else:
			x = p + 1.5
			y[n] = x*x - T2
			p = p + 1/T0
		p = p - 1

	# -- integrate
	a1 = 1 - 0.005
	z = 0
	for n in range(0,len(y)):
		y[n] = y[n] + a1 * z
		z = y[n]
		
	return -y*0.5

# -----------------------------------------------------------------------------
# 2nd order polyBLEP
#
def polyBLEP_sweep(f0,fm,fs):
	TLFO = fm/fs
	T = f0/fs
	p = 0.5
	a = 0

	y = zeros(L)
	for n in range(0,L):
		# -- sweep
		T0 = (1+a)*T
		T2 = T0+T0
		a += TLFO
		if a > 0.5:
			a = 0

		# -- polyBLEP
		y[n] = p
		if p > (1-T0):					# -- before transition
			P0 = 1/T0
			d = (p-1) * P0				# fractional phase (-1,0)		AM
			c = 0.5*d*d + d + 0.5	# correction polynomial			MMAA
			y[n] = y[n] - c			# update sample value			A
		elif p < T0:					# -- after transition
			P0 = 1/T0
			d = p * P0					# fractional phase (0,1)		M
			c = -0.5*d*d + d - 0.5	# correction polynomial			MMAA
			y[n] = y[n] - c			# update sample value			A
		y[n] = 2*y[n] - 1				#										MA

		p += T0
		if p > 1: p -= 1
	return y


# =============================================================================
# main
# =============================================================================

# -- signals
dpw = dpw_sweep(f0,fm,fs)
ptr = ptr_sweep(f0,fm,fs)
fdf = fdf_sweep(f0,fm,fs)
blep = polyBLEP_sweep(f0,fm,fs)

# -- audio files
# wavfile.write('dpw_trans.wav', fs, array(dpw*32000, dtype='int16') )
# wavfile.write('ptr_trans.wav', fs, array(ptr*32000, dtype='int16') )
# wavfile.write('fdf_trans.wav', fs, array(fdf*32000, dtype='int16') )
# wavfile.write('blep_trans.wav', fs, array(blep*32000, dtype='int16') )


# -- plotting -----------------------------------------------------------------

fig = figure(facecolor='#e2e2e2', figsize=(10,6))
font = FontProperties(family='Times New Roman', size=21)

t0 = 8725
dt = 240

p1 = subplot(221)
p1.plot(dpw, 'k', lw=1)
ylim(-1.25,4.25)
xlim(t0,t0+dt)
p1.yaxis.set_ticklabels(['', '-1.0', '0.0', '1.0', '2.0', '3.0', '4.0'])
labels = p1.get_xticklabels() + p1.get_yticklabels()
setp(labels, fontproperties=font)
xlabel('Time (samples)', fontproperties=font)
text(t0+dt/2,-3.828, '(a)', fontproperties=font)
c = Ellipse((8821,3.62), 10, 0.4, fill=False, color='black')
p1.add_artist(c)
grid()

p2 = subplot(222)
p2.plot(blep, 'k', lw=1)
ylim(-1.1,1.1)
xlim(t0,t0+dt)
labels = p2.get_xticklabels() + p2.get_yticklabels()
setp(labels, fontproperties=font)
xlabel('Time (samples)', fontproperties=font)
text(t0+dt/2,-2.15, '(b)', fontproperties=font)
grid()

fig.subplots_adjust(left=0.12, bottom=0.12, right=0.96, top=0.9, wspace=0.2, hspace=0.38)

show()
