# =============================================================================
# DPW sawtooth signals
# =============================================================================
from scipy.signal import *
from scipy import *
from pylab import *
from matplotlib.font_manager import FontProperties


f0 = 2489.		# midi note 99 (D#7)
fs = 44100.
T0 = f0/fs
P0 = fs/f0

t = arange(0,1*fs)
L = len(t)

# -----------------------------------------------------------------------------
# (1) bipolar modulo counter
#
def phi(T0,p0=0):
	s = zeros(L)
	p = p0
	for n in range(0,L):
		s[n] = p
		p += T0
		if p > 1: p -= 1
	return s

# -----------------------------------------------------------------------------
# (2) trivial saw
#
def saw(T0):
	s = 2*phi(T0,0.5) - 1
	return s

# -----------------------------------------------------------------------------
# (3) DPW N=2 : 3 multiplications + 2 additions
#
def DPW2(T0):
	z0 = 0
	c = 1/(4*T0)
	p = phi(T0,0.5)
	y = zeros(L)
	for n in range(0,L):
		s  = 2*p[n] - 1	# trivial saw	MA
		s2 = s*s				# waveshaper	M
		y0 = s2 - z0		# diff			A
		y[n] = y0 * c		# scaling		M
		z0 = s2
	y[0] = 0
	return y

# -----------------------------------------------------------------------------
# (3) DPW N=3 : 4 multiplications + 4 additions
#
def DPW3(T0):
	z0 = z1 = 0
	c = (P0*P0)/24
	p = phi(T0,0.5)
	y = zeros(L)
	for n in range(0,L):
		s  = 2*p[n] - 1	# trivial saw	MA
		s3 = s*s*s - s		# waveshaper	MMA
		y0 = s3 - z0		# diff			A
		y1 = y0 - z1		# diff			A
		y[n] = y1 * c		# scaling		M
		z0 = s3
		z1 = y0
	y[0] = y[1] = 0
	return y

# -----------------------------------------------------------------------------
# (3) DPW N=4 : 4 multiplications + 5 additions
#
def DPW4(T0):
	z0 = z1 = z2 = 0
	c = (P0*P0*P0)/192
	p = phi(T0,0.5)
	y = zeros(L)
	for n in range(0,L):
		s  = 2*p[n] - 1	# trivial saw	MA
		s2 = s*s				# waveshaper	M
		s4 = s2*(s2 - 2)	# waveshaper	MA
		y0 = s4 - z0		# diff			A
		y1 = y0 - z1		# diff			A
		y2 = y1 - z2		# diff			A		
		y[n] = y2 * c		# scaling		M
		z0 = s4
		z1 = y0
		z2 = y1
	y[0] = y[1] = y[2] = 0
	return y


# =============================================================================
# main
# =============================================================================

# -- DPW sawtooths
s0 = saw(T0)
s2 = DPW2(T0)
s3 = DPW3(T0)
s4 = DPW4(T0)


# =============================================================================
# plotting
# =============================================================================

fig = figure(figsize=(10,8), facecolor='#e2e2e2')
font = FontProperties(family='serif', size=12)
PL = 3*P0
fig.canvas.set_window_title('DPW')
suptitle('DPW sawtooth signals, $f_0$= 2489 Hz, $f_s$= 44100 Hz', fontproperties=font)

# -- helper
def set_waveplot(p,label, marker=None,stem=None,base=None):
	grid(True)
	xlim(0,PL)
	ylim(-1.1,1.1)
	xlabel('Time (samples)', fontproperties=font)
	ylabel(label, fontproperties=font, rotation=90)
	labels = p.get_xticklabels() + p.get_yticklabels()
	setp(labels, fontproperties=font)
	if marker:	setp(marker, 'markersize', 6)
	if stem:		setp(stem, 'alpha','0')
	if base:		setp(base, 'color', 'None')

# -- helper
def set_specplot(p):
	grid(True)
	xlim(0,fs/2)
	ylim(-80,6)
	xlabel('Frequency (kHz)', fontproperties=font)
	ylabel('Magnitude (dB)', fontproperties=font)
	labels = p.get_xticklabels() + p.get_yticklabels()
	setp(labels, fontproperties=font)
	p.xaxis.set_ticklabels(["0","5","10","15","20"])
	p.yaxis.set_ticks([0,-20,-40,-60,-80])

# -- magnitude spectrum
NFFT = 16384*2
NWIN = 16384
win = chebwin(NWIN, 120)
win = append(win, zeros(NFFT-NWIN))
scal = NFFT*sqrt(mean(win**2))
def spectrum(s):
	spec = fft(win*s[0:NFFT])
	mags = sqrt(spec[0:NFFT/2].real**2 + spec[0:NFFT/2].imag**2)
	norm = 20*log10(mags/scal)
	spec = norm - max(norm)
	return spec


p11 = subplot(421)
p11.plot(s0, 'k', lw=1)
marker,stem,base = p11.stem(t[0:PL], s0[0:PL], linefmt='k-', markerfmt='ko')
set_waveplot(p11,'trivial', marker,stem,base)

m0  = spectrum(s0)
p12 = subplot(422)
p12.plot(linspace(0,fs/2,len(m0)), m0, 'k', lw=1)
set_specplot(p12)

p21 = subplot(423)
p21.plot(s2, 'k', lw=1)
marker,stem,base = p21.stem(t[0:PL], s2[0:PL], linefmt='k-', markerfmt='ko')
set_waveplot(p21,'N=2', marker,stem,base)

m2  = spectrum(s2)
p22 = subplot(424)
p22.plot(linspace(0,fs/2,len(m2)), m2, 'k', lw=1)
set_specplot(p22)

p31 = subplot(425)
p31.plot(s3, 'k', lw=1)
marker,stem,base = p31.stem(t[0:PL], s3[0:PL], linefmt='k-', markerfmt='ko')
set_waveplot(p31,'N=3', marker,stem,base)

m3  = spectrum(s3)
p32 = subplot(426)
p32.plot(linspace(0,fs/2,len(m3)), m3, 'k', lw=1)
set_specplot(p32)

p41 = subplot(427)
p41.plot(s4, 'k', lw=1)
marker,stem,base = p41.stem(t[0:PL], s4[0:PL], linefmt='k-', markerfmt='ko')
set_waveplot(p41,'N=4', marker,stem,base)

m4  = spectrum(s4)
p42 = subplot(428)
p42.plot(linspace(0,fs/2,len(m4)), m4, 'k', lw=1)
set_specplot(p42)


fig.subplots_adjust(left=0.1, right=0.92, top=0.94, bottom=0.07, wspace=0.27, hspace=0.33)
show()
