# =============================================================================
# PolyBLEP  sawtooth signals
# =============================================================================
from scipy.signal import *
from scipy import *
from pylab import *
from matplotlib.font_manager import FontProperties


f0 = 2489.		# midi note 99 (D#7)
fs = 44100.
T0 = f0/fs
P0 = fs/f0

t = arange(0,1*fs)
L = len(t)

# -----------------------------------------------------------------------------
# (1) bipolar modulo counter
#
def phi(T0,p0=0):
	s = zeros(L)
	p = p0
	for n in range(0,L):
		s[n] = p
		p += T0
		if p > 1: p -= 1
	return s

# -----------------------------------------------------------------------------
# (2) trivial saw
#
def saw(T0):
	s = 2*phi(T0,0.5) - 1
	return s


# -----------------------------------------------------------------------------
# 2nd order polyBLEP
#
# operations per	period	sample
# - multiply		x		1+4T
# - add				x		1+2T
# - branch			x		1+T
#
def polyBLEP2(T0,P0):
	h = -1
	p = phi(T0,0.5)
	y = zeros(L)
	for n in range(0,L):
		y[n] = p[n]
		if p[n] > (1-T0):				# -- before transition
			d = p[n]-1+T0				# fractional phase (negative, in samples)
			d = d*P0						# fractional phase (-1,0)
			c = 0.5*d*d					# correction polynomial
			c = c * h					# scale with transition height
			y[n] = y[n] + c			# update sample value
		elif p[n] < T0:				# -- after transition
			d = p[n]						# fractional phase (positive, in samples)
			d *= P0						# fractional phase (0,1)
			c = -0.5*d*d + d - 0.5	# correction polynomial
			c = c * h					# scale with transition height
			y[n] = y[n] + c			# update sample value
		y[n] = 2*y[n] - 1
	return y

# -----------------------------------------------------------------------------
# 2nd order polyBLEP (SPM 2007)
#
# operations per	period	sample
# - multiply		P+6		1+6T
# - add				P+7		1+7T
# - branch			P+1		1+T
#
def polyBLEP2_SPM(T0,P0):
	p = phi(T0,0.5)
	y = zeros(L)
	for n in range(0,L):
		y[n] = p[n]
		if p[n] > (1-T0):				# -- before transition
			d = (p[n]-1) * P0			# fractional phase (-1,0)		AM
			c = 0.5*d*d + d + 0.5	# correction polynomial			MMAA
			y[n] = y[n] - c			# update sample value			A
		elif p[n] < T0:				# -- after transition
			d = p[n] * P0				# fractional phase (0,1)		M
			c = -0.5*d*d + d - 0.5	# correction polynomial			MMAA
			y[n] = y[n] - c			# update sample value			A
		y[n] = 2*y[n] - 1				#										MA
	return y


# =============================================================================
# main
# =============================================================================

# -- PoplyBLEP sawtooths
s0 = saw(T0)
s1 = zeros(L)
s2 = polyBLEP2_SPM(T0,P0)
s3 = zeros(L)


# =============================================================================
# plotting
# =============================================================================

fig = figure(figsize=(10,8), facecolor='#e2e2e2')
font = FontProperties(family='serif', size=12)
PL = 3*P0
fig.canvas.set_window_title('PolyBLEP')
suptitle('PolyBLEP sawtooth signals, $f_0$= 2489 Hz, $f_s$= 44100 Hz', fontproperties=font)

# -- helper
def set_waveplot(p,label, marker=None,stem=None,base=None):
	grid(True)
	xlim(0,PL)
	ylim(-1.1,1.1)
	xlabel('Time (samples)', fontproperties=font)
	ylabel(label, fontproperties=font, rotation=90)
	labels = p.get_xticklabels() + p.get_yticklabels()
	setp(labels, fontproperties=font)
	if marker:	setp(marker, 'markersize', 6)
	if stem:		setp(stem, 'alpha','0')
	if base:		setp(base, 'color', 'None')

# -- helper
def set_specplot(p):
	grid(True)
	xlim(0,fs/2)
	ylim(-80,6)
	xlabel('Frequency (kHz)', fontproperties=font)
	ylabel('Magnitude (dB)', fontproperties=font)
	labels = p.get_xticklabels() + p.get_yticklabels()
	setp(labels, fontproperties=font)
	p.xaxis.set_ticklabels(["0","5","10","15","20"])
	p.yaxis.set_ticks([0,-20,-40,-60,-80])

# -- magnitude spectrum
NFFT = 16384*2
NWIN = 16384
win = chebwin(NWIN, 120)
win = append(win, zeros(NFFT-NWIN))
scal = NFFT*sqrt(mean(win**2))
def spectrum(s):
	spec = fft(win*s[0:NFFT])
	mags = sqrt(spec[0:NFFT/2].real**2 + spec[0:NFFT/2].imag**2)
	norm = 20*log10(mags/scal)
	spec = norm - max(norm)
	return spec


p11 = subplot(421)
p11.plot(s0, 'k', lw=1)
marker,stem,base = p11.stem(t[0:PL], s0[0:PL], linefmt='k-', markerfmt='ko')
set_waveplot(p11,'trivial', marker,stem,base)

m0  = spectrum(s0)
p12 = subplot(422)
p12.plot(linspace(0,fs/2,len(m0)), m0, 'k', lw=1)
set_specplot(p12)

p21 = subplot(423)
p21.plot(s1, 'k', lw=1)
marker,stem,base = p21.stem(t[0:PL], s1[0:PL], linefmt='k-', markerfmt='ko')
set_waveplot(p21,'N=2', marker,stem,base)

m1  = spectrum(s1)
p22 = subplot(424)
p22.plot(linspace(0,fs/2,len(m1)), m1, 'k', lw=1)
set_specplot(p22)

p31 = subplot(425)
p31.plot(s2, 'k', lw=1)
marker,stem,base = p31.stem(t[0:PL], s2[0:PL], linefmt='k-', markerfmt='ko')
set_waveplot(p31,'2nd order', marker,stem,base)

m2  = spectrum(s2)
p32 = subplot(426)
p32.plot(linspace(0,fs/2,len(m2)), m2, 'k', lw=1)
set_specplot(p32)

p41 = subplot(427)
p41.plot(s3, 'k', lw=1)
marker,stem,base = p41.stem(t[0:PL], s3[0:PL], linefmt='k-', markerfmt='ko')
set_waveplot(p41,'N=4', marker,stem,base)

m3  = spectrum(s3)
p42 = subplot(428)
p42.plot(linspace(0,fs/2,len(m3)), m3, 'k', lw=1)
set_specplot(p42)


fig.subplots_adjust(left=0.1, right=0.92, top=0.94, bottom=0.07, wspace=0.27, hspace=0.33)
show()
