from scipy.signal import *
from scipy import *
from pylab import *
import pylab
from matplotlib.font_manager import FontProperties
from scipy.io import wavfile

f0 = 2490.
fs = 44100.
T0 = f0/fs
P0 = fs/f0

t = arange(0,3*fs)
L = len(t)


# =============================================================================
# SIGNALS
# =============================================================================

# -----------------------------------------------------------------------------
# Trivial is simple, but contains lots of aliasing
#
def trivial_pulse(w):
	s = zeros(L)
	p = 0
	for n in range(0,L):
		if p < w:	s[n] = w-1
		else:			s[n] = w
		p += T0
		if p > 1:
			p -= 1
	return s

# -----------------------------------------------------------------------------
# PTR algorithm is given by eq.(8), it is identical to the sawtooth case.
# Transition polynomials can be derived from Table 1, noting that x(n) is given
# by (w-1) or w, and that the pulse signal is scaled by 0.5.
#
def PTR1_pulse(w):
	s = zeros(L)		# output signal
	p = 0					# unipolar phase
	cw = 0				# no offset in this case
	x1 = w-1	- cw		# aliasing contaminated input (low)
	x2 = w	- cw		# aliasing contaminated input (high)
	for n in range(0,L):
		if p < w:		# two transitions per period
			if p < T0:	s[n] = -p*P0 + w			# = (w-1) + (-2*p*P0 + 2)/2
			else:			s[n] = x1
		else:
			pw = p-w
			if pw < T0:	s[n] = pw*P0 + w - 1;	# = w - (-2*pw*P0 + 2)/2
			else:			s[n] = x2
		p += T0
		if p > 1:
			p -= 1
	return s

# -----------------------------------------------------------------------------
#
def PTR2_pulse(w):
	T2 = 2*T0
	P2 = (P0*P0)/2
	s = zeros(L)
	p = 0
	for n in range(0,L):
		if p < w:
			if p < T0:		s[n] = -(p*p)*P2 + w							# = (w-1) + (-(p*p)*(P0*P0) + 2)/2
			elif p < T2:	s[n] =  (p*p)*P2 - 2*p*P0 + w + 1		# = (w-1) + ( (p*p)*(P0*P0) - 4*p*P0 + 4)/2
			else:				s[n] = w-1
		else:
			pw = p-w
			if pw < T0:		s[n] =  (pw*pw)*P2 + w - 1					# = w     - (-(pw*pw)*(P0*P0) + 2)/2
			elif pw < T2:	s[n] = -(pw*pw)*P2 + 2*pw*P0 + w - 2	# = w     - ( (pw*pw)*(P0*P0) - 4*pw*P0 + 4)/2
			else:				s[n] = w
		p += T0
		if p > 1:
			p -= 1
	return s

# -----------------------------------------------------------------------------
#
def PTR3_pulse(w):
	T2 = 2*T0
	T3 = 3*T0
	P2 = (P0*P0)/2
	P3 = (P0*P0*P0)/6
	s = zeros(L)
	p = 0
	for n in range(0,L):
		if p < w:
			if p < T0:		s[n] =  -(p*p*p)*P3 + w												# = (w-1) + (-((p**3)*P0**3)/3 + 2)/2
			elif p < T2:	s[n] = 2*(p*p*p)*P3 - 3*(p*p)*P2 + 1.5*p*P0 + w - 0.5		# = (w-1) + (2*((p**3)*P0**3)/3 - 3*(p**2)*P0**2 + p*3*P0 + 1)/2
			elif p < T3:	s[n] =  -(p*p*p)*P3 + 3*(p*p)*P2 - 4.5*p*P0 + w + 3.5		# = (w-1) + (-((p**3)*P0**3)/3 + 3*(p**2)*P0**2 - 9*p*P0 + 9)/2
			else:				s[n] = w-1
		else:
			pw = p - w
			if pw < T0:		s[n] =    (pw*pw*pw)*P3 + w - 1											# = w - (-((pw**3)*P0**3)/3 + 2)/2
			elif pw < T2:	s[n] = -2*(pw*pw*pw)*P3 + 3*(pw*pw)*P2 - 1.5*pw*P0 + w - 0.5	# = w - (2*((pw**3)*P0**3)/3 - 3*(pw**2)*P0**2 + pw*3*P0 + 1)/2
			elif pw < T3:	s[n] =    (pw*pw*pw)*P3 - 3*(pw*pw)*P2 + 4.5*pw*P0 + w - 4.5	# = w - (-((pw**3)*P0**3)/3 + 3*(pw**2)*P0**2 - 9*pw*P0 + 9)/2
			else:				s[n] = w
		p += T0
		if p > 1:
			p -= 1
	return s


# =============================================================================
# PWM
# =============================================================================

# -----------------------------------------------------------------------------
#
def trivial_pwm(lfoHz, lfoMax, secs):
	nsamples = secs*fs
	s = zeros(nsamples)
	w = 0.5
	dw = lfoHz / fs
	p = 0
	for n in range(0,nsamples):
		if p < w:	s[n] = w-1
		else:			s[n] = w
		w += dw
		if w > lfoMax or w < 0.5:
			dw = -dw
		p += T0
		if p > 1:
			p -= 1
	return s

# -----------------------------------------------------------------------------
#
def PTR3_pwm(lfoHz, lfoMax, secs):
	nsamples = secs*fs
	s = zeros(nsamples)
	w = 0.5
	dw = lfoHz / fs
	#
	T2 = 2*T0
	T3 = 3*T0
	P2 = (P0*P0)/2
	P3 = (P0*P0*P0)/6
	p = 0
	for n in range(0,nsamples):
		# -- PTR3
		if p < w:
			if p < T0:		s[n] =  -(p*p*p)*P3 + w
			elif p < T2:	s[n] = 2*(p*p*p)*P3 - 3*(p*p)*P2 + 1.5*p*P0 + w - 0.5
			elif p < T3:	s[n] =  -(p*p*p)*P3 + 3*(p*p)*P2 - 4.5*p*P0 + w + 3.5
			else:				s[n] = w-1
		else:
			pw = p - w
			if pw < T0:		s[n] =    (pw*pw*pw)*P3 + w - 1
			elif pw < T2:	s[n] = -2*(pw*pw*pw)*P3 + 3*(pw*pw)*P2 - 1.5*pw*P0 + w - 0.5
			elif pw < T3:	s[n] =    (pw*pw*pw)*P3 - 3*(pw*pw)*P2 + 4.5*pw*P0 + w - 4.5
			else:				s[n] = w
		p += T0
		if p > 1:
			p -= 1
		#
		w += dw
		if w > lfoMax or w < 0.5:
			dw = -dw
	return s

# -- spectrum
def spectrum(s, NFFT=16384*2, NWIN=16384):
	win = chebwin(NWIN, 120)
	win = append(win, zeros(NFFT-NWIN))
	scal = NFFT*sqrt(mean(win**2))
	spec = fft(win*s[0:NFFT])
	mags = sqrt(spec[0:NFFT/2].real**2 + spec[0:NFFT/2].imag**2)
	norm = 20*log10(mags/scal)
	spec = norm - max(norm)
	return spec


# =============================================================================
# main
# =============================================================================

# -- signals
w = 0.5
s0 = trivial_pulse(w)
s1 = PTR1_pulse(w)
s2 = PTR2_pulse(w)
s3 = PTR3_pulse(w)

spec0 = spectrum(s0)
spec1 = spectrum(s1)
spec2 = spectrum(s2)
spec3 = spectrum(s3)

# wavfile.write('trivial_square.wav', fs, array(s0*32000, dtype='int16') )
# wavfile.write('ptr_square.wav', fs, array(s3*32000, dtype='int16') )


# -- pwm
f0_pwm = 698.46
T0 = f0_pwm/fs
P0 = 1/T0
lfoHz = 0.25
lfoMax = 0.75	# 0.5 - 1.0
secs = 4
s4 = trivial_pwm(lfoHz, lfoMax, secs)
s5 = PTR3_pwm(lfoHz, lfoMax, secs)
# wavfile.write('trivial_pwm.wav', fs, array(s4*32000, dtype='int16') )
# wavfile.write('ptr_pwm.wav', fs, array(s5*32000, dtype='int16') )


# -- plotting -----------------------------------------------------------------

fig = figure(facecolor='#e2e2e2', figsize=(10,10))
font = FontProperties(family='Times New Roman', size=16)

def set_plot_props_1(p, label, marker, stem, base, PP):
	setp(marker, 'markerfacecolor', 'black')
	setp(marker, 'markersize', 8)
	setp(base, 'color', 'black')
	setp(stem, 'alpha','0')
	xlim(0,PP)
	ylim(-1.1,1.1)
	p.xaxis.set_ticks([0,10,20,30,40])
	xlabel('Time (samples)', fontproperties=font)
	ylabel(label, fontproperties=font, rotation=360)
	labels = p.get_xticklabels() + p.get_yticklabels()
	setp(labels, fontproperties=font)

def set_plot_props_2(p):
	xlim(0,fs/2)
	ylim(-80,6)
	labels = p.get_xticklabels() + p.get_yticklabels()
	setp(labels, fontproperties=font)
	p.xaxis.set_ticklabels(["0","5","10","15","20"])
	p.yaxis.set_ticks([0,-20,-40,-60,-80])
	xlabel('Frequency (kHz)', fontproperties=font)
	ylabel('Magnitude (dB)', fontproperties=font)


PP = 3*fs/f0
ti = arange(0,PP, 1./20)

p1 = subplot(421)
marker,stem,base = pylab.stem(t[0:PP+1], s0[0:PP+1], markerfmt='ko')
yi = stineman_interp(ti, t[0:PP], s0[0:PP], None)
p1.plot(ti, yi, 'k')
set_plot_props_1(p1, '(a)', marker, stem, base, PP)

p2 = subplot(422)
p2.plot(linspace(0,fs/2,len(spec0)), spec0, 'k', lw=1)
set_plot_props_2(p2)

p3 = subplot(423)
marker,stem,base = pylab.stem(t[0:PP+1], s1[0:PP+1], markerfmt='ko')
yi = stineman_interp(ti, t[0:PP], s1[0:PP], None)
p3.plot(ti, yi, 'k')
set_plot_props_1(p3, '(b)', marker, stem, base, PP)

p4 = subplot(424)
p4.plot(linspace(0,fs/2,len(spec1)), spec1, 'k', lw=1)
set_plot_props_2(p4)

p5 = subplot(425)
marker,stem,base = pylab.stem(t[0:PP+1], s2[0:PP+1], markerfmt='ko')
yi = stineman_interp(ti, t[0:PP], s2[0:PP], None)
p5.plot(ti, yi, 'k')
set_plot_props_1(p5, '(c)', marker, stem, base, PP)

p6 = subplot(426)
p6.plot(linspace(0,fs/2,len(spec2)), spec2, 'k', lw=1)
set_plot_props_2(p6)

p7 = subplot(427)
marker,stem,base = pylab.stem(t[0:PP+1], s3[0:PP+1], markerfmt='ko')
yi = stineman_interp(ti, t[0:PP], s3[0:PP], None)
p7.plot(ti, yi, 'k')
set_plot_props_1(p7, '(d)', marker, stem, base, PP)

p8 = subplot(428)
p8.plot(linspace(0,fs/2,len(spec3)), spec3, 'k', lw=1)
set_plot_props_2(p8)

fig.subplots_adjust(left=0.13, bottom=0.12, right=0.99, top=0.9, wspace=0.3, hspace=0.38 )
show()
