# =============================================================================
# BLIT-FDF sawtooth signals (Nam et al. 2010)
# B-spline interpolation
# =============================================================================
from scipy.signal import *
from scipy import *
from pylab import *
from matplotlib.font_manager import FontProperties


f0 = 2489.		# midi note 99 (D#7)
fs = 44100.
T0 = f0/fs
P0 = fs/f0

t = arange(0,1*fs)
L = len(t)

# -----------------------------------------------------------------------------
# (1) bipolar modulo counter
#
def phi(T0,p0=0):
	s = zeros(L)
	p = p0
	for n in range(0,L):
		s[n] = p
		p += T0
		if p > 1: p -= 1
	return s

# -----------------------------------------------------------------------------
# (2) trivial saw
#
def saw(T0):
	s = 2*phi(T0,0.5) - 1
	return s

# -----------------------------------------------------------------------------
# Leaky integrator
#
# operations per	period	sample
# - multiply		P			1
# - add				P			1
# - branch			0			0
#
def leaky_integrator(x, a1, z0=0):
	z = z0
	y = zeros(len(x))
	for n in range(0,len(x)):
		y[n] = x[n] + a1 * z
		z = y[n]
	return y

# -----------------------------------------------------------------------------
# 1st order B-spline/Lagrange interpolator
#
# operations per	period	sample
# - multiply		P+2		1+2T
# - add				P+2		1+2T
# - branch			P+1		1+T
#
def BLIT_Bspline1(P0,T0):
	p = 0.5 * P0
	T2 = T0+T0
	y = zeros(L)

	# -- create BLITs
	for n in range(0,L):
		if p >= 1:		y[n] = -T2
		elif p >= 0:	y[n] = 2 - 2*p - T2		# AM
		else:
			y[n] = 2 + 2*p - T2						# AM
			p = p + P0
		p = p - 1

	# -- integrate
	e = 0.005
	y = leaky_integrator(y, 1-e)		# MA
	return -y

# -----------------------------------------------------------------------------
# 2nd order B-spline interpolator
#
# operations per	period	sample
# - multiply		P+4		1+4T
# - add				P+5		1+5T
# - branch			P+2		1+2T
#
def BLIT_Bspline2(P0,T0):
	p = 0.5 * P0
	T2 = T0+T0
	y = zeros(L)

	# -- create BLITs
	for n in range(0,L):
		if p >= 1.5:
			y[n] = -T2						# dc offset
		elif p >= 0.5:
			x = p - 1.5						# A
			y[n] = x*x - T2				# MA
		elif p >= -0.5:
			y[n] = 1.5 - 2*p*p - T2		# AMM
		else:
			x = p + 1.5						# A
			y[n] = x*x - T2				# MA
			p = p + P0
		p = p - 1

	# -- integrate
	e = 0.005
	y = leaky_integrator(y, 1-e)		# MA
	return -y

# -----------------------------------------------------------------------------
# 3rd order B-spline interpolator
#
# operations per	period	sample
# - multiply		P+10		1+10T
# - add				P+8		1+8T
# - branch			P+3		1+3T
#
def BLIT_Bspline3(P0,T0):
	p = 0.5 * P0
	T2 = T0+T0
	y = zeros(L)

	# -- create BLITs
	for n in range(0,L):
		if p >= 2:
			y[n] = -T2								# dc offset
		elif p >= 1:
			x = 2-p									# A
			y[n] = x*x*x/3. - T2					# MMMA
		elif p >= 0:
			y[n] =  p*p*(p-2) + 4/3. - T2		# MMAA
		elif p >= -1:
			y[n] = -p*p*(p+2) + 4/3. - T2		# MMAA
		else:
			x = 2+p									# A
			y[n] = x*x*x/3. - T2					# MMMA
			p = p + P0
		p = p - 1

	# -- integrate
	e = 0.005
	y = leaky_integrator(y, 1-e)				# MA
	return -y


# =============================================================================
# main
# =============================================================================

# -- PTR sawtooths
s0 = saw(T0)
s1 = BLIT_Bspline1(P0,T0)
s2 = BLIT_Bspline2(P0,T0)
s3 = BLIT_Bspline3(P0,T0)


# =============================================================================
# plotting
# =============================================================================

fig = figure(figsize=(10,8), facecolor='#e2e2e2')
font = FontProperties(family='serif', size=12)
PL = 3*P0
fig.canvas.set_window_title('BLIT-FDF')
suptitle('BLIT-FDF sawtooth signals, $f_0$= 2489 Hz, $f_s$= 44100 Hz', fontproperties=font)

# -- helper
def set_waveplot(p,label, marker=None,stem=None,base=None):
	grid(True)
	xlim(0,PL)
	ylim(-1.1,1.1)
	xlabel('Time (samples)', fontproperties=font)
	ylabel(label, fontproperties=font, rotation=90)
	labels = p.get_xticklabels() + p.get_yticklabels()
	setp(labels, fontproperties=font)
	if marker:	setp(marker, 'markersize', 6)
	if stem:		setp(stem, 'alpha','0')
	if base:		setp(base, 'color', 'None')

# -- helper
def set_specplot(p):
	grid(True)
	xlim(0,fs/2)
	ylim(-80,6)
	xlabel('Frequency (kHz)', fontproperties=font)
	ylabel('Magnitude (dB)', fontproperties=font)
	labels = p.get_xticklabels() + p.get_yticklabels()
	setp(labels, fontproperties=font)
	p.xaxis.set_ticklabels(["0","5","10","15","20"])
	p.yaxis.set_ticks([0,-20,-40,-60,-80])

# -- magnitude spectrum
NFFT = 16384*2
NWIN = 16384
win = chebwin(NWIN, 120)
win = append(win, zeros(NFFT-NWIN))
scal = NFFT*sqrt(mean(win**2))
def spectrum(s):
	spec = fft(win*s[0:NFFT])
	mags = sqrt(spec[0:NFFT/2].real**2 + spec[0:NFFT/2].imag**2)
	norm = 20*log10(mags/scal)
	spec = norm - max(norm)
	return spec


p11 = subplot(421)
p11.plot(s0, 'k', lw=1)
marker,stem,base = p11.stem(t[0:PL], s0[0:PL], linefmt='k-', markerfmt='ko')
set_waveplot(p11,'trivial', marker,stem,base)

m0  = spectrum(s0)
p12 = subplot(422)
p12.plot(linspace(0,fs/2,len(m0)), m0, 'k', lw=1)
set_specplot(p12)

p21 = subplot(423)
p21.plot(s1, 'k', lw=1)
marker,stem,base = p21.stem(t[0:PL], s1[0:PL], linefmt='k-', markerfmt='ko')
set_waveplot(p21,'N=2', marker,stem,base)

m1  = spectrum(s1)
p22 = subplot(424)
p22.plot(linspace(0,fs/2,len(m1)), m1, 'k', lw=1)
set_specplot(p22)

p31 = subplot(425)
p31.plot(s2, 'k', lw=1)
marker,stem,base = p31.stem(t[0:PL], s2[0:PL], linefmt='k-', markerfmt='ko')
set_waveplot(p31,'N=3', marker,stem,base)

m2  = spectrum(s2)
p32 = subplot(426)
p32.plot(linspace(0,fs/2,len(m2)), m2, 'k', lw=1)
set_specplot(p32)

p41 = subplot(427)
p41.plot(s3, 'k', lw=1)
marker,stem,base = p41.stem(t[0:PL], s3[0:PL], linefmt='k-', markerfmt='ko')
set_waveplot(p41,'N=4', marker,stem,base)

m3  = spectrum(s3)
p42 = subplot(428)
p42.plot(linspace(0,fs/2,len(m3)), m3, 'k', lw=1)
set_specplot(p42)


fig.subplots_adjust(left=0.1, right=0.92, top=0.94, bottom=0.07, wspace=0.27, hspace=0.33)
show()
