function [reconstructed_feas1, reconstructed_uct1] = tcmi(feas1, mask1, gmm_model1, feamin1)

% TCMI truncated conditional mean imputation presented in (Gonzalez et al. 2013)
%
% Input:
%
% feas1      compressed spectral features (channels x frames)
% mask1      missing-data mask (channels x frames) reliable/unreliable 1/0
% feamin1    minimum feature value (1x1)
%
% Clean speech model:
%
% gmm_model1.mu (dim x num components)
% gmm_model1.sigma (dim x dim x num components)
% gmm_model1.weight (num components x 1)
%
% Output:
%
% reconstructed_feas1  approximate posterior mean (channels x frames)
% reconstructed_uct1   approximate posterior variance (channels x frames)

% (C) 2014 Ulpu Remes
% MIT license
% For license terms and references, see README.txt

[feadim1, N1]=size(feas1);
[modeldim1, M1]=size(gmm_model1.mu); 

assert(modeldim1==feadim1,'Feature dimension does not match model dimension');

% initialise variables

pos1=zeros(1,M1);
mixture_feas1=zeros(feadim1,M1);
mixture_uct1=zeros(feadim1,M1);
reconstructed_feas1=zeros(feadim1,N1);
reconstructed_uct1=zeros(feadim1,N1);

for n=1:N1
   
   unreliable_feas1 = find(mask1(:,n)==0);
   reliable_feas1 = find(mask1(:,n)==1);
  
   % reliable frames need not be reconstructed:
   
   if length(reliable_feas1)==feadim1
       reconstructed_feas1(:,n)=feas1(:,n);
       reconstructed_uct1(:,n)=zeros(feadim1,1);
       continue;
   end
   
   % reorganisation matrix for mixed frames
   
   eye1 = eye(feadim1);
   eye1 = eye1([reliable_feas1; unreliable_feas1],:);
     
    % FEATURE RECONSTRUCTION

    for m = 1:M1 

        reorganised_feature=eye1*feas1(:,n);

        % compute mean and cov for unreliable data

        sigma1 = gmm_model1.sigma(:,:,m);
        sigma1 = eye1*sigma1*eye1';

        mu1 = gmm_model1.mu(:,m);
        mu1 = eye1*mu1;

        % define sr1, su1, sru1 etc.

        num_reliable1 = length(reliable_feas1);

        sr1 = sigma1(1:num_reliable1,1:num_reliable1);
        sm1 = sigma1(1:num_reliable1,num_reliable1+1:feadim1);
        sm2 = sigma1(num_reliable1+1:feadim1,1:num_reliable1);
        su1 = sigma1(num_reliable1+1:feadim1,num_reliable1+1:feadim1);

        mr2 = mu1(1:num_reliable1);
        mu2 = mu1(num_reliable1+1:feadim1);

        % marginal distribution for the unreliable components

        su3 = su1-sm2*pinv(sr1)*sm1;
        mu3 = mu2+sm2*pinv(sr1)*(reorganised_feature(1:num_reliable1)-mr2);

        %%%% cluster posterior
        
        % reliable components
        
        if num_reliable1 > 0
           
            pos_reliable = mvnpdf(reorganised_feature(1:num_reliable1),mr2,sr1);
        else
            pos_reliable = 1;
        end
        
        % unreliable components
        
        cum_dist=zeros(feadim1-num_reliable1,1);
        for k=1:feadim1-num_reliable1
            cum_dist(k)=normcdf(reorganised_feature(k+num_reliable1),mu3(k),sqrt(su3(k,k)));
        end
        pos1(m)=gmm_model1.weight(m)*pos_reliable*prod(cum_dist)+exp(-700);

        %%%% feature estimate
        
        ubm=(reorganised_feature(num_reliable1+1:feadim1)-mu3)./max(sqrt(diag(su3)),exp(-700));
        lbm = (feamin1-mu3)./max(sqrt(diag(su3)),exp(-700));
        lbm = max(lbm,-exp(100));
        
        normubm=normpdf(ubm);
        normlbm=normpdf(lbm);
        denominator=max(normcdf(ubm)-normcdf(lbm),exp(-700));
        
        reorganised_feature(num_reliable1+1:feadim1)=mu3+sqrt(diag(su3)).*(normlbm-normubm)./denominator;
        resu=diag(su3).*(1+(lbm.*normlbm-ubm.*normubm)./denominator-((normlbm-normubm)./denominator).^2);
        
        mixture_feas1(:,m) = eye1'*reorganised_feature;
        mixture_uct1(:,m)=eye1'*[zeros(num_reliable1,1); max(resu,0)];

    end

    reconstructed_feas1(:,n)=sum(bsxfun(@times,mixture_feas1,pos1/sum(pos1)),2);
    reconstructed_uct1(:,n)=sum(bsxfun(@times,mixture_uct1+power(bsxfun(@minus,mixture_feas1,reconstructed_feas1(:,n)),2),pos1/sum(pos1)),2);

end